/*
 * ImgAge.cpp
 *
 *  Created on: 2009-12-15
 *      Author: Wojciech Waga <wojciech.waga.com>
 */

#include "ImageProperties.hpp"
#include "Image.hpp"
#include "ImgAge.hpp"
#include "../Lattice.hpp"


ImgAge::ImgAge(const ImageProperties &p, const Lattice &l):
    Image(p,l)
{
  height=prop.getValueInt("Aheight");
  yfactor=prop.getValueInt("Ayfactor");
  gd_male=prop.getValueInt("Amales");
  gd_female=prop.getValueInt("Afemales");
}

void ImgAge::write(const std::string & filename) const
{
  std::vector <usint> pop;
  for (usint i=0; i<lat.atab.size(); i++)
    pop.push_back(i);
  write(filename,pop,false);
}

void ImgAge::write(const std::string & filename, bool f) const
{
  std::vector <usint> pop;
  for (usint i=0; i<lat.atab.size(); i++)
    pop.push_back(i);
  write(filename,pop,f);
}

void ImgAge::write(const std::string & filename, usint population) const
{
  std::vector <usint> pop(1,population);
  write(filename,pop,false);
}

void ImgAge::write(const std::string & filename, usint population, bool f) const
{
  std::vector <usint> pop(1,population);
  write(filename,pop,f);
}

void ImgAge::write(const std::string & filename, const std::vector<usint> &popNum) const
{
  write(filename,popNum,false);
}

void ImgAge::write(const std::string & filename, const std::vector<usint> &popNum, bool f) const
{
  const uint life_span=lat.atab[popNum[0]].getPopulation()->getLifeSpan();

  std::vector<uint> tabM(life_span,0);
  std::vector<uint> tabF(life_span,0);

  for (std::vector<usint>::const_iterator it=popNum.begin(); it!=popNum.end(); ++it)
    {
    const Atab &atab=lat.atab[*it];

    if (atab.getPopulation()->getLifeSpan()!=life_span)
      throw std::runtime_error("TxtAge: You have to select populations with the same life spans for cumulative data.");

    if (!f)
      {
      for (uint i=0; i<life_span; i++)
        {
        tabM[i]+=atab.getAtabInd(i,MALE);
        tabF[i]+=atab.getAtabInd(i,FEMALE);
        }
      }
    else
      {
        for (uint i=0; i<life_span; i++)
        {
        tabM[i]+=atab.getAtabInd(i,FIVETed_MALE);
        tabF[i]+=atab.getAtabInd(i,FIVETed_FEMALE);
        }
      }
    }


  const uint width=yfactor*life_span;
  gdImagePtr im = gdImageCreateTrueColor(width, height);

  uint gd_color_male  =gdImageColorAllocateAlpha(im, (gd_male&0xff0000)>>16 , (gd_male&0x00ff00)>>8 , gd_male&0x0000ff , 0);
  uint gd_color_female=gdImageColorAllocateAlpha(im, (gd_female&0xff0000)>>16 , (gd_female&0x00ff00)>>8 , gd_female&0x0000ff , 50);

  {
    uint gd_black=gdImageColorAllocate(im,0,0,0);
    gdImageRectangle(im,0,0,width-1,height-1,gd_black);
    gdImageColorDeallocate(im,gd_black);
  }
  //count maximum number of individuals at certain age
  uint max=1;
  for (uint age=0; age<life_span; age++)
    {
      if(tabM[age]>max)
        max=tabM[age];
      if(tabF[age]>max)
        max=tabF[age];
    }

   for (uint i=0; i<life_span*yfactor; i++)
     {
        uint sumM=(height*tabM[i/yfactor])/max;
        uint sumF=(height*tabF[i/yfactor])/max;

        if (sumM==0)
          if(tabM[i/yfactor]) sumM=2;

        if (sumF==0)
          if(tabF[i/yfactor]) sumF=2;



       if (i%yfactor || (yfactor==1))
         {
           if (sumM) gdImageLine(im,i,height-sumM,i,height-1,gd_color_male);
           if (sumF) gdImageLine(im,i,height-sumF,i,height-1,gd_color_female);
         }
     }

   {
     uint gd_red=gdImageColorAllocate(im,255,0,0);

     for (std::vector<usint>::const_iterator it=popNum.begin(); it!=popNum.end(); ++it)
       gdImageLine(im,lat.atab[*it].getPopulation()->getReproductionAge()*yfactor,0,lat.atab[*it].getPopulation()->getReproductionAge()*yfactor,height-1,gd_red); //reproductive age line
     gdImageColorDeallocate(im,gd_red);
   }

   FILE * pngout = fopen((filename+".png").c_str(), "wb");
   gdImagePng(im, pngout);
   fclose(pngout);
   gdImageDestroy(im);
}
