/*
 * imgChromosome.cpp
 *
 *  Created on: 2009-11-29
 *      Author: Wojciech Waga <wojciech.waga.com>
 */


#include <vector>
#include <iostream>
#include <boost/lexical_cast.hpp>

#include "ImageProperties.hpp"
#include "Image.hpp"
#include "../detail/Tab.hpp"
#include "ImgChromosome.hpp"
#include "../Lattice.hpp"

ImgChromosome::ImgChromosome(const ImageProperties &p, const Lattice &l):
    Image(p,l), color(p.getValueVectInt("Scolor"))
{
  height=prop.getValueInt("Cheight");
  yfactor=prop.getValueInt("Cyfactor");
  fgcolor=prop.getValueInt("Cfgcolor");
  bgcolor=prop.getValueInt("Cbgcolor");
}

void ImgChromosome::write(const std::string & filename) const
{
  write(filename, 0, false);
}

void ImgChromosome::write(const std::string & filename, bool f) const
{
  write(filename, 0, f);
}

void ImgChromosome::write(const std::string & filename, usint population) const
{
  write(filename, population, false);
}

void ImgChromosome::write(const std::string & filename, usint population, bool f) const
{
  const Population *pop=lat.atab[population].getPopulation().get();

  uint nc=pop->getNumChunks();
  if (pop->hasYchr())
    nc+=pop->getChromLength(pop->getNumChromosomes()-1);

  const uint width=nc*64;

  gdImagePtr im = gdImageCreateTrueColor(width*yfactor,height);
  int gd_red=gdImageColorAllocate(im,255,0,0);
  int gd_green=gdImageColorAllocate(im,0,255,0);

  if (im==0)
    {
      std::cerr << "Could not allocate chromosome image (size: " << width*yfactor << ',' << height << ")\n";
      throw std::runtime_error("If you want collective chromosome chart, all populations must have the same layout and genome length.");
    }

  int gd_white=gdImageColorAllocate(im,255,255,255);
  gdFontPtr font = gdFontGetGiant();
  std::string chromosome="chr: ";

  uint offset=0;
  for (usint i=0; i<pop->getNumChromosomes()-1; i++)
    {
      const uint factor=yfactor*(pop->getChromLength(i)*64)/pop->getLifeSpan();
      drawSingleChromosomeChart(im,offset,i,pop,f);
      gdImageLine(im,offset,0,offset,height-1,gd_white);
      gdImageLine(im,offset+1,0,offset+1,height-1,gd_white);
      for (uint j=1; j<=yfactor; j++)
	{
	  gdImageLine(im,offset+j+pop->getBirthAge()*factor,0,offset+j+pop->getBirthAge()*factor,height-1,gd_green);
	  gdImageLine(im,offset+j+pop->getReproductionAge()*factor,0,offset+j+pop->getReproductionAge()*factor,height-1,gd_red);
	}
      gdImageString(im,font,offset+5,5,(unsigned char*)(chromosome+boost::lexical_cast<std::string>(i+1)).c_str(),gd_white);
      offset+=pop->getChromLength(i)*64*yfactor;
    }

  if (!pop->hasYchr())
    {
      const uint i=pop->getNumChromosomes()-1;
      const uint factor=yfactor*(pop->getChromLength(i)*64)/pop->getLifeSpan();
      drawSingleChromosomeChart(im,offset,i,pop,f);
      gdImageLine(im,offset,0,offset,height-1,gd_white);
      gdImageLine(im,offset+1,0,offset+1,height-1,gd_white);
      for (uint j=1; j<=yfactor; j++)
	{
	  gdImageLine(im,offset+j+pop->getBirthAge()*factor,0,offset+j+pop->getBirthAge()*factor,height-1,gd_green);
	  gdImageLine(im,offset+j+pop->getReproductionAge()*factor,0,offset+j+pop->getReproductionAge()*factor,height-1,gd_red);
	}
      gdImageString(im,font,offset+5,5,(unsigned char*)(chromosome+boost::lexical_cast<std::string>(i+1)).c_str(),gd_white);
    } else
    {
      const uint factor=yfactor*(pop->getChromLength(pop->getNumChromosomes()-1)*64)/pop->getLifeSpan();
      drawXChromosomeChart(im,offset,pop,f);
      gdImageLine(im,offset,0,offset,height-1,gd_white);
      gdImageLine(im,offset+1,0,offset+1,height-1,gd_white);
      for (uint j=1; j<=yfactor; j++)
	{
	  gdImageLine(im,offset+j+pop->getBirthAge()*factor,0,offset+j+pop->getBirthAge()*factor,height-1,gd_green);
	  gdImageLine(im,offset+j+pop->getReproductionAge()*factor,0,offset+j+pop->getReproductionAge()*factor,height-1,gd_red);
	}
      gdImageString(im,font,offset+5,5,(unsigned char*)(chromosome+" X").c_str(),gd_white);
      offset+=pop->getChromLength(pop->getNumChromosomes()-1)*64*yfactor;
      drawYChromosomeChart(im,offset,pop,f);
      gdImageLine(im,offset,0,offset,height-1,gd_white);
      gdImageLine(im,offset+1,0,offset+1,height-1,gd_white);
      for (uint j=1; j<=yfactor; j++)
	{
	  gdImageLine(im,offset+j+pop->getBirthAge()*factor,0,offset+j+pop->getBirthAge()*factor,height-1,gd_green);
	  gdImageLine(im,offset+j+pop->getReproductionAge()*factor,0,offset+j+pop->getReproductionAge()*factor,height-1,gd_red);
	}
      gdImageString(im,font,offset+5,5,(unsigned char*)(chromosome+" Y").c_str(),gd_white);

    }
  //coloring bars
  int color_r= gdImageColorAllocateAlpha(im,255,  0,  0, 80);
  int color_g= gdImageColorAllocateAlpha(im,  0,255,  0, 80);
  int color_b= gdImageColorAllocateAlpha(im,  0,  0,255, 80);


  for (uint i=0; i<color.size(); ++i)
    {
      const int pos=(color[i]*64+20)*yfactor;
      gdImageFilledRectangle(im,pos+( 0*yfactor),0,pos+( 7*yfactor),height,color_r);
      gdImageFilledRectangle(im,pos+( 8*yfactor),0,pos+(15*yfactor),height,color_g);
      gdImageFilledRectangle(im,pos+(16*yfactor),0,pos+(23*yfactor),height,color_b);
      gdImageString(im,font,pos+( 8*yfactor),height-15,(unsigned char*)(boost::lexical_cast<std::string>(i)).c_str(),gd_red);
    }

  gdImageColorDeallocate(im,gd_red);
  gdImageColorDeallocate(im,gd_green);
  gdImageColorDeallocate(im,color_r);
  gdImageColorDeallocate(im,color_g);
  gdImageColorDeallocate(im,color_b);

  FILE * pngout = fopen((filename+".png").c_str(), "wb");
  gdImagePng(im, pngout);
  fclose(pngout);
  gdImageDestroy(im);
}

void ImgChromosome::drawSingleChromosomeChart(gdImagePtr im, const uint offset, const usint chr, const Population * pop, bool f) const
{
  const uint width=pop->getChromLength(chr)*64;

  const Tab &tab=lat.getLattice();
  std::vector<uint> sum(width,0);

  const uint chr_offset=pop->getChromOffset(chr);
  uint count=0;

  if (!f)
  {
    for (uint i=0; i<tab.getHeight(); ++i)
      for (uint j=0; j<tab.getWidth(); ++j)
        if (tab.isAlive(j,i) && (lat.getPopulation(tab.get(j,i).getPop()).get()==pop))
          {
	    count+=2;
	    for (uint k=0; k<pop->getChromLength(chr); k++)
	      for (int y=0; y<64; y++)
	        {
		  if ((1ULL<<(63-y))&(tab.get(j,i).h1[k+chr_offset]))
		    sum[y+64*k]++;
		  if ((1ULL<<(63-y))&(tab.get(j,i).h2[k+chr_offset]))
		    sum[y+64*k]++;
	        }
          }
  }
  else
  {
    for (uint i=0; i<tab.getHeight(); ++i)
      for (uint j=0; j<tab.getWidth(); ++j)
        if ((tab.isAlive(j,i)) && (lat.getPopulation(tab.get(j,i).getPop()).get()==pop) && (tab.get(j,i).isFIVETed()))
          {
            count+=2;
            for (uint k=0; k<pop->getChromLength(chr); k++)
              for (int y=0; y<64; y++)
                {
                  if ((1ULL<<(63-y))&(tab.get(j,i).h1[k+chr_offset]))
                    sum[y+64*k]++;
                  if ((1ULL<<(63-y))&(tab.get(j,i).h2[k+chr_offset]))
                    sum[y+64*k]++;
                }
          }
  }  

  const uint max=count;
  const uint factor=yfactor*(pop->getChromLength(chr)*64)/pop->getLifeSpan();
  drawChart(im,offset,sum,max,factor);
}

void ImgChromosome::drawXChromosomeChart(gdImagePtr im, const uint offset, const Population * pop, bool f) const
{
  const int chr=pop->getNumChromosomes()-1;
  const uint width=pop->getChromLength(chr)*64;

  const Tab &tab=lat.getLattice();
  std::vector<uint> sum(width,0);

  const uint chr_offset=pop->getChromOffset(chr);
  uint count=0;

  if (!f)
  {
    for (uint i=0; i<tab.getHeight(); ++i)
      for (uint j=0; j<tab.getWidth(); ++j)
        if (tab.isAlive(j,i) && (lat.getPopulation(tab.get(j,i).getPop()).get()==pop))
          {
	    if(tab.get(j,i).getGender()==FEMALE)
	      {//2 chromosomes for FEMALES
	        count+=2;
	        for (uint k=0; k<pop->getChromLength(chr); k++)
		  for (int y=0; y<64; y++)
		    {
		      if ((1ULL<<(63-y))&(tab.get(j,i).h1[k+chr_offset]))
		        sum[y+64*k]++;
		      if ((1ULL<<(63-y))&(tab.get(j,i).h2[k+chr_offset]))
		        sum[y+64*k]++;
		    }
	      } else
              {//1 chromosome for MALES
	        count++;
	        for (uint k=0; k<pop->getChromLength(chr); k++)
		  for (int y=0; y<64; y++)
		    {
		      if (!tab.get(j,i).getY())
		        {
			  if ((1ULL<<(63-y))&(tab.get(j,i).h2[k+chr_offset]))
			    sum[y+64*k]++;
		        }
		      else
		        {
			  if ((1ULL<<(63-y))&(tab.get(j,i).h1[k+chr_offset]))
			    sum[y+64*k]++;
		        }
		    }
              }
          }
  }
  else
  {
    for (uint i=0; i<tab.getHeight(); ++i)
      for (uint j=0; j<tab.getWidth(); ++j)
        if (tab.isAlive(j,i) && (lat.getPopulation(tab.get(j,i).getPop()).get()==pop) && tab.get(j,i).isFIVETed())
          {
            if(tab.get(j,i).getGender()==FEMALE)
              {//2 chromosomes for FEMALES
                count+=2;
                for (uint k=0; k<pop->getChromLength(chr); k++)
                  for (int y=0; y<64; y++)
                    {
                      if ((1ULL<<(63-y))&(tab.get(j,i).h1[k+chr_offset]))
                        sum[y+64*k]++;
                      if ((1ULL<<(63-y))&(tab.get(j,i).h2[k+chr_offset]))
                        sum[y+64*k]++;
                    }
              } else
              {//1 chromosome for MALES
                count++;
                for (uint k=0; k<pop->getChromLength(chr); k++)
                  for (int y=0; y<64; y++)
                    {
                      if (!tab.get(j,i).getY())
                        {
                          if ((1ULL<<(63-y))&(tab.get(j,i).h2[k+chr_offset]))
                            sum[y+64*k]++;
                        }
                      else
                        {
                          if ((1ULL<<(63-y))&(tab.get(j,i).h1[k+chr_offset]))
                            sum[y+64*k]++;
                        }
                    }
              }
          }
  }
  const uint max=count;
  const uint factor=yfactor*(pop->getChromLength(chr)*64)/pop->getLifeSpan();
  drawChart(im,offset,sum,max,factor);
}

void ImgChromosome::drawYChromosomeChart(gdImagePtr im, const uint offset, const Population * pop, bool f) const
{
  const int chr=pop->getNumChromosomes()-1;
  const uint width=pop->getChromLength(chr)*64;

  const Tab &tab=lat.getLattice();
  std::vector<uint> sum(width,0);

  const uint chr_offset=pop->getChromOffset(chr);
  uint count=0;

  if (!f)
  {
    for (uint i=0; i<tab.getHeight(); ++i)
      for (uint j=0; j<tab.getWidth(); ++j)
        if (tab.isAlive(j,i) && (lat.getPopulation(tab.get(j,i).getPop()).get()==pop) && tab.get(j,i).getGender()==MALE)
          {
          count++;
          for (uint k=0; k<pop->getChromLength(chr); k++)
            for (int y=0; y<64; y++)
              {
                if (tab.get(j,i).getY())
                  {
                    if ((1ULL<<(63-y))&(tab.get(j,i).h2[k+chr_offset]))
                      sum[y+64*k]++;
                  }
                else
                  {
                    if ((1ULL<<(63-y))&(tab.get(j,i).h1[k+chr_offset]))
                      sum[y+64*k]++;
                  }
              }
          }
  }
  else
  {
    for (uint i=0; i<tab.getHeight(); ++i)
      for (uint j=0; j<tab.getWidth(); ++j)
        if (tab.isAlive(j,i) && (lat.getPopulation(tab.get(j,i).getPop()).get()==pop) && tab.get(j,i).getGender()==MALE  && tab.get(j,i).isFIVETed())
          {
          count++;
          for (uint k=0; k<pop->getChromLength(chr); k++)
            for (int y=0; y<64; y++)
              {
                if (tab.get(j,i).getY())
                  {
                    if ((1ULL<<(63-y))&(tab.get(j,i).h2[k+chr_offset]))
                      sum[y+64*k]++;
                  }
                else
                  {
                    if ((1ULL<<(63-y))&(tab.get(j,i).h1[k+chr_offset]))
                      sum[y+64*k]++;
                  }
              }
          }
  }

  const uint max=count;
  const uint factor=yfactor*(pop->getChromLength(chr)*64)/pop->getLifeSpan();
  drawChart(im,offset,sum,max,factor);
}

void ImgChromosome::drawChart(gdImagePtr im, const uint offset, std::vector<uint> sum, uint max, uint factor) const
{
  //std::cerr << "IMG: (" << sum[0] << ' ' << max << ')' <<  std::endl;
  const uint width=sum.size();
  for (uint i=0; i<width*yfactor; i++)
    {
    const uint val=(i%yfactor==0 && (yfactor!=1))?0:uint((height-1)*(sum[i/yfactor]/float(max)));

    for (uint j=height-1; j!=0; j--)
      im->tpixels[j][i+offset]=bgcolor;

    for (uint j=height-1; (j>(height-val-1)) &&( j!=0); j--)
        im->tpixels[j][i+offset]=fgcolor;
    }

  { //yellow horizontal lines 0 25% 50% 75% 1
    int gd_yellow=gdImageColorAllocate(im,255,255,0);
    gdImageLine(im,offset,0,offset+ width*yfactor-1,0,gd_yellow);
    gdImageLine(im,offset,1*height/4,offset+width*yfactor-1,1*height/4,gd_yellow);
    gdImageLine(im,offset,2*height/4,offset+width*yfactor-1,2*height/4,gd_yellow);
    gdImageLine(im,offset,3*height/4,offset+width*yfactor-1,3*height/4,gd_yellow);
    gdImageLine(im,offset,height-1,offset+width*yfactor-1,height-1,gd_yellow);
    gdImageColorDeallocate(im, gd_yellow);
  }
}
