/*
 * ImgLD.cpp
 *
 *  Created on: 2011-01-05
 *      Author: Wojciech Waga <wojciech.waga.com>
 */

#include <sstream>
#include <iomanip>
#include <boost/lexical_cast.hpp>

#include "../Lattice.hpp"
#include "ImageProperties.hpp"
#include "Image.hpp"
#include "ImgLD.hpp"
#include "../Population.hpp"

//todo zmenic wszystko na korzystanie z lattice::width lattice::height

ImgLD::ImgLD(const ImageProperties &p, const Lattice &l):
  Image(p,l)
{
  const Population *pop=lat.atab[0].getPopulation().get();
  nc=pop->getNumChunks();
  if (pop->hasYchr())
    nc+=pop->getChromLength(pop->getNumChromosomes()-1);
  width=nc*64+50;
  height=width;
}

void ImgLD::write(const std::string & fn) const
{
  const Population *pop=lat.atab[0].getPopulation().get();

  for (uint i=0; i<lat.atab.size(); i++)
    if (!pop->sameLayout(lat.atab.at(i).getPopulation().get()))
      throw std::runtime_error("If you want collective chromosome chart, all populations must have the same layout and genome length.");

  gdImagePtr im = gdImageCreateTrueColor(width,height);
  int gd_white=gdImageColorAllocate(im,255,255,255);
  int gd_blue=gdImageColorAllocate(im,0,255,255);

  gdImageLine(im,50,0,50,height,gd_white);
  gdImageLine(im,0,50,width,50,gd_white);

  const Tab &tab=lat.getLattice();
  uint latticeHeight=tab.getHeight();
  uint latticeWidth=tab.getWidth();

  std::vector<uint> alleles(nc*64,0);
  float count=0;

  for (uint i=0; i<latticeWidth; i++)
    for (uint j=0; j<latticeHeight; j++)
      if (tab.isAlive(i,j))
        {
          count++;
          for (usint c=0; c<nc; c++)
            {
              const uint64_t h1=tab.get(j,i).getH1(c);
              for (usint y=0; y<64; y++)
                if ((1ULL<<(63-y))&h1) alleles[c*64+y]++;
            }
        }

  for (uint i=0; i<nc*64; i++)
    {
      float height=(alleles[i]/count)*50;
      gdImageLine(im,i+50,50,i+50,50-height,gd_blue);
      gdImageLine(im,50,i+50,50-height,i+50,gd_blue);
    }
  //TODO: add X/Y chromosome support

  std::vector< std::vector<float> > LD(nc*64, std::vector<float>(nc*64,0));

  for (uint i=0; i<latticeWidth; i++)
    for (uint j=0; j<latticeHeight; j++)
      if (tab.isAlive(i,j))
        for (usint c1=0; c1<nc; c1++)
          for (usint c2=0; c2<nc; c2++)
            {
              const uint64_t h1a=tab.get(j,i).getH1(c1);
              const uint64_t h1b=tab.get(j,i).getH1(c2);
              for (usint y1=0; y1<64; y1++)
                for (usint y2=0; y2<64; y2++)
                  if (((1ULL<<(63-y1))&h1a) && ((1ULL<<(63-y2))&h1b))
                    LD[c1*64+y1][c2*64+y2]++;
            }

  for (int i=0; i<nc*64; i++)
    for (int j=0; j<nc*64; j++)
      {
        const float p1=(alleles[i]/count);
        const float p2=(alleles[j]/count);
        if (p1>=0.99 || p2>=0.99)
          {
            im->tpixels[i+50][j+50]=0x444444;
            continue;
          }

        LD[i][j]=LD[i][j]/count-p1*p2; //D
        LD[i][j]/=sqrt(p1*p2*(1-p1)*(1-p2)); //r
        LD[i][j]*=LD[i][j]; //r^2

        //std::cout << LD[i][j] <<' ';

        im->tpixels[i+50][j+50]=uint(LD[i][j]*255)<<8;
      }

  FILE *pngout = fopen(fn.c_str(), "wb");
  gdImagePng(im, pngout);
  fclose(pngout);
  gdImageDestroy(im);
}

