/*
 * Lattice.cpp
 *
 *  Created on: 2009-12-30
 *      Author: Wojciech Waga <wojciech.waga.com>
 */

#include "Lattice.hpp"


namespace //assertion helpers
{

bool checkMarkers(const Tab  & tab, Lattice::PopulationPtr pop)
{
  for (uint i=0; i<tab.getHeight(); i++)
    for (uint j=0; j<tab.getWidth(); j++)
      {
	if (!tab.isAlive(j,i))
	  continue;
	const Specimen &s=tab.get(j,i);
	const usint num_chromosomes=pop->getNumChromosomes();
	const usint num_autosomes=pop->hasYchr()?num_chromosomes-1:num_chromosomes;

	for (uint i_chr=0, offset=0; i_chr<num_autosomes; i_chr++)
	  {
	    if (pop->hasRecognition(i_chr))
	      {
		const usint bitnum=pop->getMarkerBit(i_chr);
		const uint64_t cx =getBit(bitnum%64);
		const uint cxc=bitnum/64+offset;

		if ((s.h1[cxc]|s.h2[cxc])&cx)
		  {
		    //everything's fine
		  } else
		  {
		    std::cerr << "cxc:" << cxc << " bitnum: " << (bitnum) << std::endl;
		    print_binary(s.h1[cxc]); std::cout << std::endl;
		    print_binary(s.h2[cxc]); std::cout << std::endl;
		    print_binary(cx); std::cout << std::endl;
		    return false;
		  }
	      }
	    offset+=pop->getChromLength(i_chr);
	  }
      }
  return true;
}

};

Lattice::Lattice(usint w, usint h):
  tab_(w,h),
  gennum(0),capacity_(uint(w)*uint(h)),
  width(w),height(h),positions1(0),
  max_father_dist(0),max_child_dist(0),
  positions2(0), panmixia(false), empty(0),
  empty_ind(0), altered(true), F(capacity_),
  stats_period_(10000)
{

}

Lattice::~Lattice()
{
  if(positions1)
    delete[] positions1;
  if(positions2)
    delete[] positions2;
}

void Lattice::kill(usint x, usint y)
{
  assert(latticeCount()==atabCount());
  const Specimen &s=tab_.get(x,y);
  const Gender g=s.getGender();
  const uint age=s.getAge();
  const usint pop=s.getPop();

  if (atab[pop].kill(tab_,x,y,g,age))
    {
      tab_.setAlive(x,y,false);
      if (panmixia)
        panPutFreePlace(x,y);
    }

  assert(latticeCount()==atabCount());
}

void Lattice::killSquare(usint x, usint y, usint w, usint h)
{
  assert(latticeCount()==atabCount());
  for (uint i=x; i<x+w; i++)
    for (uint j=y; j<y+h; j++)
      kill(i,j);
  assert(latticeCount()==atabCount());
}

void Lattice::fill(boost::shared_ptr<Population> p)
{
  fill(p,0,0,width,height);
}

const Specimen & Lattice::getGuy(usint x, usint y) const
{
  return tab_.get(x%getWidth(),y%getHeight());
}

bool Lattice::isEmpty(usint x, usint y) const
{
  return !tab_.isAlive(x,y);
}

void Lattice::purge(float f)
{
  uint toKill=static_cast<uint>(f*latticeCount());

  uint killed=0;

  while(killed<toKill)
    {
      usint x=rand()%getWidth();
      usint y=rand()%getWidth();
      if (tab_.isAlive(x,y))
        {
          killed++;
          kill(x,y);
        }
    }
}

void Lattice::fill(PopulationPtr p, usint x, usint y, usint w, usint h)
{
  if (isPopNew(p))
    registerPop(p);

  if (x+w>width || y+h>height)
    throw NchrException("Fill coordinates exceed size of the lattice!");

  p->read();

  const uchar nr=getPopNum(p);

  if (w==2 && h==1)
    {//if there are only two individuals make sure they are man and woman
      Specimen &s1=tab_.get(x,y);
      Specimen &s2=tab_.get(x+1,y);
      if (tab_.isAlive(x,y))
	kill(x,y);
      if (tab_.isAlive(x+1,y))
	kill(x+1,y);
      s1.initialize(*p,Gender(true));
      s2.initialize(*p,Gender(false));
      s1.setPop(nr);
      s2.setPop(nr);
      tab_.setAlive(x,y,true);
      tab_.setAlive(x+1,y,true);
      atab[nr].atabInsert(y,x,s1);
      atab[nr].atabInsert(y,x+1,s2);
    } else
    {//if there are more than two, gender is random
      for (uint j=y; j<(y+h); j++)
	for (uint i=x; i<(x+w); i++)
	  {
	    Specimen &s=tab_.get(i,j);
	    if (tab_.isAlive(i,j))
	      kill(i,j);
	    s.initialize(*p);
	    s.setPop(nr);
	    tab_.setAlive(i,j,true);
	    atab[nr].atabInsert(j,i,s);
	  }
    }

  if (panmixia==true)
    {
      //reload free places
      setPanmixia(false);
      setPanmixia(true);
    }

  assert(latticeCount()==atabCount());
  //assert(checkMarkers());
  for (uint i=0; i<pop_numbers.size(); i++)
    atab[i].updateIterators();
}


void Lattice::nexGen(bool frozen)
{
  update();

  for (uint i=0; i<pop_numbers.size(); i++)
    atab[i].updateIterators();

  assert(latticeCount()==atabCount());

  if (altered)
    {
      if (panmixia==true)
	{
	  //reload free places
	  setPanmixia(false);
	  setPanmixia(true);
	}
      altered=false;
    }

  gennum++;

  if (!frozen)
    scrollAtab();

  Specimen frozen_child; //for frozen populations

  F.clear();

  for (uint p=0; p<pop_numbers.size(); p++)
    F.insert(atab[p].begin(), atab[p].end());

  while(!F.isEmpty())
    {
      const Point motherpos=F.getFemale();
      const Specimen & mother=tab_.get(motherpos);
      const usint p=mother.getPop();
      const Population *pop=pop_numbers[p].second.get();

      if (mother.isInfertile() && !pop->hasFivet())
        continue; //there is no FIVET and mother is infertile, this is a shortcut for performance only

      assert(mother.getGender()==FEMALE);

      const usint bpoisson=pop->getBPoisson();

      for (uint brood=0; brood < bpoisson; brood++)
        {
          Point child_pl;

          if (!frozen)
            child_pl=isPanmictic()? findPlaceAnywhere() : findPlace(motherpos.second,motherpos.first,p);

          if (child_pl.first!=0xffff)
            {
              assert(frozen || !tab_.isAlive(child_pl));

              const Specimen * father=isPanmictic()? findFatherAnywhere(p) : findFather(motherpos.second,motherpos.first,p);

              if (father)
                {
                  assert(father->getAge()>=pop->getReproductionAge());
                  assert(father->getAge()<pop->getLifeSpan());

                  bool with_fivet=false;
                  //when one partner is infertile then the couple is infertile
                  const bool couple_infertile=father->isInfertile() || mother.isInfertile();

                  if (couple_infertile)
                    {
                      if (!pop->hasFivet())
                        {
                          if (isPanmictic() && !frozen)
                            panPutFreePlace(child_pl.first,child_pl.second);
                          continue; //couple is infertile and there is no FIVET
                        }

                      if (pop->getFIVETefficacy() < rand_0to1())
                        {
                          if (isPanmictic() && !frozen)
                            panPutFreePlace(child_pl.first,child_pl.second);
                          continue; //couple is infertile and FIVET is ineffective
                        }

                      with_fivet=true;
                    }

                  Specimen & child= frozen?frozen_child:tab_.get(child_pl);

                  putChild(child,mother,*father,with_fivet);

                  atab[p].statsConception();

                  if (frozen)
                    {
                      const uint death_year=predictDeathAge(child,child_pl.first,child_pl.second);
                      atab[p].statsLifeExpectancy(death_year);
                    }

                  if(atab[p].isLoser(child,child_pl.first,child_pl.second,0,env.get()))
                    {
                      if (isPanmictic() && !frozen)
                        panPutFreePlace(child_pl.first,child_pl.second);
                    }
                  else
                    {
                      atab[p].statsBirth();
                      if (!frozen)
                        {
                          tab_.setAlive(child_pl,true);
                          atab[p].atabInsert(child_pl.second,child_pl.first,child);
                        }
                    }
                }
              else
                break; //no father in range
            }
          else
            break; //no child in range
        }
    }
  assert(latticeCount()==atabCount());

  uint spec_num=0;
  for (uint p=0; p<pop_numbers.size(); p++)
    spec_num+=atab[p].getSpecimenCount();
  if (spec_num==0)
    {
      std::string s="Empty lattice in generation no " + boost::lexical_cast<std::string>(gennum);
      throw NchrException(s.c_str());
    }

  if (gennum%stats_period_==0)
    for (uint p=0; p<pop_numbers.size(); p++)
      atab[p].commitStats(getLattice(),gennum);
}


Point  Lattice::findPlace(usint y, usint x,usint pop)
{
  const int child_dist=pop_numbers[pop].second->getChildDist();
  int xmin=x-child_dist,xmax=x+child_dist,ymin=y-child_dist,ymax=y+child_dist;
  int index=0;
  int imod, jmod;

  for (int i=ymin; i<=ymax; i++)
    for (int j=xmin; j<=xmax; j++)
      {
        imod=(i+height)%height;
        jmod=(j+width)%width;

        if (!tab_.isAlive(jmod,imod))
          {
            positions2[index].first=jmod;
            positions2[index].second=imod;
            index++;
          }
      }

  if (index)
    return positions2[rand()%index];

  return std::make_pair(0xffff,0xffff);
}

Point  Lattice::findPlaceAnywhere()
{
  if (empty_ind==0)
    return std::make_pair(0xffff,0xffff);
  const uint num=rand()%empty_ind;
  const Point ret=empty[num];
  empty[num]=empty[--empty_ind];
  return ret;
}


const Specimen * Lattice::findFather(usint y, usint x, uchar pop)
{
  const int father_dist=pop_numbers[pop].second->getFatherDist();
  const bool promiscous=pop_numbers[pop].second->isPromisc();

  int xmin=x-father_dist,xmax=x+father_dist,ymin=y-father_dist,ymax=y+father_dist;
  uint index=0;
  int imod, jmod;

  for (int i=ymin; i<=ymax; i++)
    for (int j=xmin; j<=xmax; j++)
      {
        imod=(i+height)%height;
        jmod=(j+width)%width;

        if (tab_.isAlive(jmod,imod))
          if (tab_.get(jmod,imod).getGender()==MALE &&
              (tab_.get(jmod,imod).getAge()>pop_numbers[pop].second->getReproductionAge()) &&
              (promiscous || tab_.get(jmod,imod).getPop()==pop))
            {
              positions1[index]=&(tab_.get(jmod,imod));
              index++;
            }
      }
  if (index)
    return positions1[rand()%index];
  return NULL;
}

/**
 * Promiscuous mode is not perfect, since population number is chosen at first
 * and it's possible to find noone where there are still some men from other pops.
 */
const Specimen * Lattice::findFatherAnywhere(uchar pop)
{
  const bool promiscous=pop_numbers[pop].second->isPromisc();
  if (promiscous)
    pop=rand()%pop_numbers.size();
  Point father=atab[pop].getFather();

  if (father.second==0xffff)
    return NULL;

  return &(tab_.get(father));
}

void Lattice::scrollAtab()
{
  for (uint i=0; i<pop_numbers.size(); i++)
    atab[i].scroll(tab_,*this);
}

uchar Lattice::getPopNum(PopulationPtr p)
{
  for (std::vector< std::pair<std::string,PopulationPtr> >::iterator it=pop_numbers.begin(); it!=pop_numbers.end(); ++it)
    if (it->first.compare(p->getName())==0)
      return uchar(it-pop_numbers.begin());
  throw NchrException("No population with given name");
}

bool Lattice::isPopNew(PopulationPtr p)
{
  for (std::vector< std::pair<std::string,PopulationPtr> >::iterator it=pop_numbers.begin(); it!=pop_numbers.end(); ++it)
    if (it->first.compare(p->getName())==0)
      {
        if (it->second.get()==p.get())
          return false;
        else
          throw NchrException("Different population with the same name exists!");
      }
  return true;
}


usint Lattice::registerPop(PopulationPtr p)
{
  pop_numbers.push_back(std::make_pair(p->getName(),p));
  atab.push_back(p);

  if(max_father_dist<p->getFatherDist())
    {
      max_father_dist=p->getFatherDist();
      delete[] positions1;
      positions1=new Specimen*[(2*max_father_dist+1)*(2*max_father_dist+1)];
    }

  if(max_child_dist<p->getChildDist())
    {
      max_child_dist=p->getChildDist();
      delete[] positions2;
      positions2=new Point [(2*max_child_dist+1)*(2*max_child_dist+1)];
    }
  return pop_numbers.size();
}

uint Lattice::latticeCount() const
{
  uint result=0;
  for (uint y=0; y<height; y++)
    for (uint x=0; x<width; x++)
      if (tab_.isAlive(x,y))
        result++;
  return result;
}

uint Lattice::atabCount() const
{
  uint spec_num=0;
  for (uint p=0; p<pop_numbers.size(); p++)
    spec_num+=atab[p].getSpecimenCount();
  return spec_num;
}

uint Lattice::atabCount(usint pop, bool f) const
{
  return atab[pop].getSpecimenCount(f);
}

std::vector<bool> Lattice::computeChromosomeLayout(const Population *pop, bool formale) const
{
  const usint num_chunks=pop->getNumChunks();
  const usint num_chromosomes=pop->getNumChromosomes();

  std::vector<bool> layout(num_chunks,false);

  for (uint i=0, offset=0; i<num_chromosomes; i++)
    {
      bool flip=((rand()%2)==1);
      if (formale)
        flip=true;
      for (uint j=0; j<pop->getChromLength(i); j++)
        layout[j+offset]=flip;

      offset+=pop->getChromLength(i);
    }
  return layout;
}

void Lattice::applyMutations(uint64_t * h1, uint64_t * h2, uint nummut, usint chrom_length, usint offset, int marker) const
{
  for (uint mut=0; mut<nummut; ++mut)
    {
      const uint mut_pos=rand()%(chrom_length*64);
      const uint cx=mut_pos%64;
      const uint cxc=mut_pos/64;

      if (static_cast<uint>(marker)==mut_pos) //if marker == -1 then there is no marker
        continue; //no mutation this time

      const uint64_t bit=getBit(cx);
      h1[cxc+offset]|=bit;
      h2[cxc+offset]|=bit;
    }
}

void Lattice::applyRecombinations(uint64_t * h1, uint64_t * h2, uint numrek, usint chrom_length, usint offset) const
{
  for (uint r=0; r<numrek; r++)
    {
      const uint cx=rand()%64;
      const uint cxc=(rand()%chrom_length)+offset;
      const uint64_t cut_pos=mask64(cx+1);

      uint64_t tmp=h1[cxc]&cut_pos;
      h1[cxc]|=cut_pos;
      h1[cxc]&=(~cut_pos)|(cut_pos&h2[cxc]);
      h2[cxc]|=cut_pos;
      h2[cxc]&=(~cut_pos)|(cut_pos&tmp);

      for (uint j=cxc+1; j<(chrom_length+offset); j++)
        {
          tmp=h1[j];
          h1[j]=h2[j];
          h2[j]=tmp;
        }
    }
}

void Lattice::selectChromosome(uint64_t * Fh,uint64_t * h1, uint64_t * h2, usint chrom_length, usint offset, usint bitnum) const
{
  const uint64_t cx =getBit(bitnum%64);
  const uint cxc=bitnum/64+offset;

  if ((Fh[cxc]^h1[cxc])&cx)
    return; //One of chromosomes has a marker and one doesn't. There is nothing to do.
  else
    for (usint i=offset; i<offset+chrom_length; i++)
      h1[i]=h2[i];
}

void Lattice::putChild(Specimen & child, const Specimen & mother, const Specimen & father, bool with_fivet)
{
  const Population *pop=pop_numbers[mother.getPop()].second.get();
  const usint num_chunks=pop->getNumChunks();
  const usint num_chromosomes=pop->getNumChromosomes();
  const usint num_autosomes=pop->hasYchr()?num_chromosomes-1:num_chromosomes;

  assert(mother.getAge()>=pop->getReproductionAge());
  assert(father.getAge()>=pop->getReproductionAge());

  if (child.getNumChunks()!=num_chunks)
    child.realloc(num_chunks);

  uint64_t *Fh1=child.h1;
  uint64_t *Fh2=new uint64_t[num_chunks];

  uint64_t *Mh1=child.h2;
  uint64_t *Mh2=new uint64_t[num_chunks];

  //compute layout of chromosomes for male and female
  std::vector<bool> chr_layoutM=computeChromosomeLayout(pop,false);
  std::vector<bool> chr_layoutF=computeChromosomeLayout(pop,false);

  //***             phase 1: copy parents' genomes

  for (uint i=0; i<num_chunks; i++)
    {
      Mh1[i]=chr_layoutM[i]?father.h2[i]:father.h1[i];
      Mh2[i]=chr_layoutM[i]?father.h1[i]:father.h2[i];

      Fh1[i]=chr_layoutF[i]?mother.h2[i]:mother.h1[i];
      Fh2[i]=chr_layoutF[i]?mother.h1[i]:mother.h2[i];
    }

  //***             phase 2: add mutations (using poisson distribution)

  for (uint i_chr=0, offset=0; i_chr<num_chromosomes; i_chr++)
    {
      const usint nummutF=with_fivet?pop->mutations_increased_.getNum(i_chr):pop->mutations_.getNum(i_chr);
      const usint nummutM=with_fivet?pop->mutations_increased_.getNum(i_chr):pop->mutations_.getNum(i_chr);

      if (pop->hasRecognition(i_chr))
        {
          applyMutations(Fh1,Fh2,nummutF,pop->getChromLength(i_chr),offset,pop->getMarkerBit(i_chr));
          applyMutations(Mh1,Mh2,nummutM,pop->getChromLength(i_chr),offset,pop->getMarkerBit(i_chr));
        }
      else
        {
          applyMutations(Fh1,Fh2,nummutF,pop->getChromLength(i_chr),offset);
          applyMutations(Mh1,Mh2,nummutM,pop->getChromLength(i_chr),offset);
        }

      offset+=pop->getChromLength(i_chr);
    }


  //***             phase 3: add recombinations (using poisson distribution)

  for (uint i_chr=0, offset=0; i_chr<num_chromosomes; i_chr++)
    {
      const usint numrekF=pop->recombinations_.getNum(i_chr);
      applyRecombinations(Fh1,Fh2,numrekF,pop->getChromLength(i_chr),offset);
      offset+=pop->getChromLength(i_chr);
    }

  for (uint i_chr=0, offset=0; i_chr<num_autosomes; i_chr++)
    {
      const usint numrekM=pop->recombinations_.getNum(i_chr);
      applyRecombinations(Mh1,Mh2,numrekM,pop->getChromLength(i_chr),offset);
      offset+=pop->getChromLength(i_chr);
    }


  //***             phase 4: select chromosomes for gamete

  //if true we got Y chromosome and we are male
  bool whichY=chr_layoutM[num_chunks-1];

  for (uint i_chr=0, offset=0; i_chr<num_autosomes; i_chr++)
    {
      if (pop->hasRecognition(i_chr))
        selectChromosome(Fh1,Mh1,Mh2,pop->getChromLength(i_chr),offset,pop->getMarkerBit(i_chr));
      offset+=pop->getChromLength(i_chr);
    }


  if (whichY)
    {
      child.setY(true);
      child.setGender(MALE);
    }
  else
    child.setGender(FEMALE);

  child.setAge(pop->getBirthAge());
  child.setPop(mother.getPop());

  if (pop->hasYchr()==false)
    child.setGender(Gender(rand()%2));

  if (with_fivet)
    child.setInfertile(pop->getInfertilityAF()>rand_0to1()); //set infertility for the newborn
  else
    child.setInfertile(pop->getInfertility()>rand_0to1()); //set infertility for the newborn

  if (with_fivet)
    child.setFIVETed(true); //marks child as born after in vitro

  delete[] Mh2;
  delete[] Fh2;
}

bool Lattice::isObstacled(usint x, usint y, usint j, usint i) const
{/*
   int W,Wx,Wy;
   for (int ind=0; ind<numObstacles; ind++)
   {
   W=obstacles[2][ind]*(y-i)-obstacles[3][ind]*(x-j);
   Wx=(x-obstacles[0][ind])*(y-i)-(y-obstacles[1][ind])*(x-j);
   Wy=obstacles[2][ind]*(y-obstacles[1][ind])-obstacles[3][ind]*(x-obstacles[0][ind]);

   if (W==0) continue;

   if (W<0)
   {
   if (Wx>=W && Wx<=0 && Wy>=W && Wy<=0) return true;
   }
   else                   {
   if (Wx<=W && Wx>=0 && Wy<=W && Wy>=0) return true;
   }
   }*/
  return false;
  //todo dodac
}


void Lattice::update()
{
  usint maxF=0, maxC=0;
  for (std::vector< std::pair<std::string,PopulationPtr> >::iterator it=pop_numbers.begin(); it!=pop_numbers.end(); ++it)
    {
      if (((it->second)->getChildDist())>maxC) maxF=(it->second)->getChildDist();
      if (((it->second)->getFatherDist())>maxF) maxC=(it->second)->getFatherDist();
    }

  if(max_father_dist<maxF)
    {
      max_father_dist=maxF;
      delete[] positions1;
      positions1=new Specimen*[(2*max_father_dist+1)*(2*max_father_dist+1)];
    }

  if(max_child_dist<maxC)
    {
      max_child_dist=maxC;
      delete[] positions2;
      positions2=new Point [(2*max_child_dist+1)*(2*max_child_dist+1)];
    }
}

usint Lattice::addObstacle(std::string type, usint x, usint y, usint w, usint z)
{
  if (type.compare("line")==0)
    {
      obstacles.push_back(Line(x,y,w,z));
      return obstacles.size()-1;
    } else
    if (type.compare("box")==0)
      {
        addObstacle("line",x,y,x+w,y);
        addObstacle("line",x+w,y,x+w,y+z);
        addObstacle("line",x+w,y+z,x,y+z);
        return addObstacle("line",x,y+z,x,y);
      }
  return 0;
}

void Lattice::delObstacle(usint num)
{
  //todo
}

void Lattice::setEnvironment(Environment &e)
{
  env.reset(e.clone());
}

uint Lattice::predictDeathAge(const Specimen & s, usint x, usint y) const //spr
{
  const usint p=s.getPop();
  const Population *pop=pop_numbers[p].second.get();

  uint upto=pop->getLifeSpan();

  for (uint i=0; i<upto; i++) //TODO: change O(n) to O(log n)
    if(atab[p].isLoserAbs(s,x,y,i,env.get()))
      return i;

  return pop->getLifeSpan();
}

void Lattice::unsetEnvironment()
{
  env.reset();
}

const Environment * Lattice::getEnvironment() const
{
  return env.get();
}

void Lattice::setPanmixia(bool p)
{
  if(panmixia==p)
    return;
  panmixia=p;

  empty_ind=0;

  if (p)
    {
      empty=new Point [capacity_];
      for (usint i=0; i<tab_.getHeight(); ++i)
        for (usint j=0; j<tab_.getWidth(); ++j)
          if (!tab_.isAlive(j,i))
            panPutFreePlace(j,i);
    } else
    delete[] empty;
}

bool Lattice::isPanmictic() const
{
  return panmixia;
}

void Lattice::panPutFreePlace(uint x, uint y)
{
  assert(empty_ind<=capacity_);
  assert(!tab_.isAlive(x,y));
  empty[empty_ind].first=x;
  empty[empty_ind].second=y;
  empty_ind++;
}

void Lattice::setStatsPeriod(uint gen)
{
  stats_period_=gen;
}

void Lattice::copy(const Lattice & L, usint src_x, usint src_y, usint w, usint h, usint dst_x, usint dst_y)
{
  if (src_x+w>L.getWidth()) w=L.getWidth()-src_x;
  if (src_y+h>L.getHeight()) h=L.getHeight()-src_y;

  if (dst_x+w>getWidth()) w=getWidth()-dst_x;
  if (dst_y+h>getHeight()) h=getHeight()-dst_y;

  for (usint i=0; i<w; i++)
    for (usint j=0; j<h; j++)
      {
        if (L.isEmpty(src_x+i,src_y+j))
          continue;
        const Specimen &s=L.getGuy(src_x+i,src_y+j);
        PopulationPtr p=L.getPopulation(s.getPop());

        if (isPopNew(p))
          registerPop(p);

        usint nr=getPopNum(p);

        const usint dst_x2=dst_x+i;
        const usint dst_y2=dst_y+j;

        if (tab_.isAlive(dst_x2,dst_y2))
          kill(dst_x2,dst_y2);

        Specimen & s2=tab_.get(dst_x2,dst_y2);
        s2=s;
        s2.setPop(nr);
        tab_.setAlive(dst_x2,dst_y2,true);
        atab[nr].atabInsert(dst_y2,dst_x2,s2);
      }
}

bool Lattice::operator==(const Lattice & l2)
{
  if (gennum!=l2.gennum) return false;
  if (width!=l2.width) return false;
  if (height!=l2.height) return false;
  if (panmixia!=l2.panmixia) return false;
  if (stats_period_!=l2.stats_period_) return false;
  if (max_child_dist!=l2.max_child_dist) return false;
  if (max_father_dist!=l2.max_father_dist) return false;
  if (obstacles!=l2.obstacles) return false;

  for (uint i=0; i<pop_numbers.size(); i++)
    {
      if (pop_numbers[i].first!=l2.pop_numbers[i].first)
        return false;
      if (*(pop_numbers[i].second)!=*(l2.pop_numbers[i].second))
        return false;
    }

  return true;
}

std::vector<uint> Lattice::getStats(const std::string & pop)
{
  PopulationPtr p=getPopulation(pop);
  uint pop_num=getPopNum(p);
  return atab[pop_num].getPopStats().getLifeExpectancy();
}
