/*
 * Lattice.hpp
 *
 *  Created on: 2009-12-30
 *      Author: Wojciech Waga <wojciech.waga.com>
 */

#ifndef LATTICE_HPP_
#define LATTICE_HPP_

#include <vector>
#include <memory>
#include <iostream>
#include <boost/serialization/vector.hpp>
#include <boost/serialization/shared_ptr.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/lexical_cast.hpp>

#include "detail/auto_ptr_serial.hpp"
#include "detail/defs.hpp"
#include "detail/NchrException.hpp"
#include "detail/Tab.hpp"
#include "detail/Atab.hpp"
#include "detail/Line.hpp"
#include "detail/FemaleRandomizer.hpp"

#include "detail/Statistics.hpp"

#include "Environment.hpp"
#include "Population.hpp"
#include "Specimen.hpp"


class Lattice: public boost::noncopyable
{
public:
  typedef boost::shared_ptr<Population> PopulationPtr;

  Lattice(usint w, usint h);
  //todo check if name is unique
  void fill(PopulationPtr  p);
  void fill(PopulationPtr  p, usint x, usint y, usint w, usint h);

  /**
   * @brief Advance to next generation.
   * @param freezed if population is freezed (that is children are
   * not placed on the lattice, there is no genetic death.
   */
  void nexGen(bool freezed=false);

  /**
   * @brief Kill individual at the specified position.
   * If the field is empty function doen't do anything.
   * @param x x coordinate
   * @param y y coordinate
   */
  void kill(usint x, usint y);

  /**
   * @brief Kill all individuals in the specified rectangle.
   * @param x top left x coordinate
   * @param y top left y coordinate
   * @param w width
   * @param h height
   */
  void killSquare(usint x, usint y, usint w, usint h);

  /**
   * @brief Kill indivudals at random if lattice ocupation is over x
   * @param x - maximal occupation of the lattice
   */
  void purge(float x);

  /**
   * @brief Find free place for a child in the neighbourhood.
   * Finds only free places which are withing search area specific to
   * a given population around an individual.
   * Distance at which place is looked for.
   * @param y y coordinate of a center of the area
   * @param x x coordinate of a center of the area
   * @param pop number of population
   * @return Coordinates of a free field of the lattice within given distance.
   */
  Point  findPlace(usint y, usint x, usint pop);

  /**
   * @brief Finds a place on the whiole lattice.
   * @return Coordinates of a free field of the lattice.
   */
  Point  findPlaceAnywhere();

  /**
   * @brief Look for partner within a specified radius.
   * @param y female's y position
   * @param x female's x position
   * @param pop population from which we want to chose a partner.
   * @return  Pointer to partner or NULL.
   */

  const Specimen * findFather(usint y, usint x, uchar pop);

  /**
   * @brief Looks for male partner on the whole lattice.
   * @param pop population from which we want to chose a partner.
   * @return Pointer to partner or NULL.
   */
  const Specimen * findFatherAnywhere(uchar pop);

  /**
   * @brief Get specified guy.
   * @param x coordinate
   * @param y coordinate
   * @return Reference to a Specimen object. If field is "empty" reference to trashy object is return.
   */

  const Specimen & getGuy(usint x, usint y) const;

  /**
   * @brief Check whether or not given field is empty.
   * @param x coordinate
   * @param y coordinate
   * @return true if field is empty
   */
  bool isEmpty(usint x, usint y) const;

  uint predictDeathAge(const Specimen & s, usint x, usint y) const;

  void putChild(Specimen & child, const Specimen & mother, const Specimen & father, bool isFertile);

  /**
   * @brief copy individuals between lattices
   * Copies individuals which are inside of the speficied rectangle. You have to specify both coordinates
   * of a rectangle you want to copy and coordinates where to paste those individuals on a new lattice.
   * If there is not enough place on a new lattice excessive individuals are not to be copied. All individuals
   * that are on a new lattice prior to copying will be lost.
   * Keep in mind that population names should be unique! If they are not newly copied individuals will
   * be joined to an existing population instead a new one.
   *
   * @param L Lattice to copy individuals from
   * @param src_x top-left coordinate of a source rectangle
   * @param src_y top-left coordinate of a source rectangle
   * @param w width of rectangles
   * @param h height of rectabgles
   * @param dst_x top-left coordinate of a destination rectangle
   * @param dst_y top-left coordinate of a destination rectangle
   */
  void copy(const Lattice & L, usint src_x, usint src_y, usint w, usint h, usint dst_x, usint dst_y);

  /**
   * @brief Lists all registered populations.
   */
  const std::vector< std::pair<std::string, PopulationPtr> > listPopulations() const
  {
    return pop_numbers;
  }

  /**
   * @brief Convert population's name to a pointer.
   * @param s Name of the population to get.
   * @return Pointer to desired population.
   */
  PopulationPtr getPopulation(const std::string & s)
  {
    for (std::vector< std::pair<std::string, PopulationPtr> >::const_iterator it=pop_numbers.begin(); it!=pop_numbers.end(); ++it)
      if (s.compare(it->first)==0)
        return it->second;
    throw NchrException("Unknown population name!");
  }

  /**
   * @brief Convert population number to a pointer.
   * @param n Number of the population to get.
   * @return Pointer to desired population
   */
  PopulationPtr getPopulation(uchar n) const
    {
      return pop_numbers.at(n).second;
    }

  /**
   * @brief Set the number of generation to collect statistics.
   * @param gen - number of generations
   */
  void setStatsPeriod(uint gen);

  /**
   * @brief Gets statistics of life expectancy of newborns.
   * These statistics are collected only when lattice is frozen
   * @param pop - name of the population to collect statistics
   */
  std::vector<uint> getStats(const std::string & pop);

  void scrollAtab();

  ~Lattice();

  const Tab  & getLattice() const
  {
    return tab_;
  }

  uint getGennum() const {return gennum;}

  usint getWidth() const {return width;}
  usint getHeight() const{return height;}

  /**
   * @brief Sets panmictic behaviour on the lattice.
   */
  void setPanmixia(bool p);

  /**
   * @brief checks if a lattice is panmictic.
   */
  bool isPanmictic() const;

  void panPutFreePlace(uint x, uint y);

  std::string getPopName(uchar p) const
  {
    return pop_numbers[p].first;
  }

  //private:
  usint registerPop(PopulationPtr p);
  uchar getPopNum(PopulationPtr p);
  bool isPopNew(PopulationPtr p);

  /**
   * @brief Get the lattice's occupation
   * @return number of living individuals according to lattice
   */
  uint latticeCount() const;

  /**
     * @brief Get the atab occupation
     * @return number of living individuals according to atab
  */
  uint atabCount() const;
  uint atabCount(usint pop, bool f) const;

  /**
   * @brief Check if registered populations have changed.
   */
  void update();

  void setEnvironment(Environment &e);
  void unsetEnvironment();
  const Environment * getEnvironment() const;

  friend class boost::serialization::access;

  template<class Archive>
  void serialize(Archive & ar, const unsigned int version)
  {
    ar & gennum;
    ar & max_father_dist;
    ar & max_child_dist;
    ar & tab_;
    ar & obstacles;
    ar & panmixia;
    ar & stats_period_;
    boost::serialization::split_member(ar, *this, version);
  }

  template<class Archive>
  void load(Archive & ar, const unsigned int version)
  {
    uint pop_numbers_size;
    ar >> pop_numbers_size;

    for (uint i=0; i<pop_numbers_size; i++)
      {
        std::string name;
        PopulationPtr ptr;
        ar >> name;
        ar >> ptr;
        pop_numbers.push_back(std::make_pair(name,ptr));
      }

    altered=true;

    for (uint i=0; i<pop_numbers.size(); i++)
      atab.push_back(pop_numbers[i].second);

    for (uint y=0; y<height; y++)
      for (uint x=0; x<width; x++)
        if (tab_.isAlive(x,y))
          atab[tab_.get(x,y).getPop()].atabInsert(y,x,tab_.get(x,y));

    if (positions1)
      delete[] positions1;
    positions1=new Specimen*[(2*max_father_dist+1)*(2*max_father_dist+1)];
    if (positions2)
      delete[] positions2;
    positions2=new Point [(2*max_child_dist+1)*(2*max_child_dist+1)];
  }

  template<class Archive>
  void save(Archive & ar, const unsigned int version) const
  {
    uint pop_numbers_size=pop_numbers.size();
    ar << pop_numbers_size;

    for (uint i=0; i<pop_numbers_size; i++)
      {
        std::string name=pop_numbers[i].first;
        PopulationPtr ptr=pop_numbers[i].second;
        ar << name;
        ar << ptr;
      }

    //todo environment
    //ar & env;
  }

  friend class TxtAge;
  friend class TxtChromosome;
  friend class TxtChrStat;
  friend class ImgAge;
  friend class ImgLD;
  friend class PopCNT;
  friend class ImgChromosome;
  friend class Content;
  friend class DataExtractorAge;

  usint addObstacle(std::string type, usint x, usint y, usint w, usint z);
  void delObstacle(usint num);

  const std::vector<Line> & getObstacles() const
  {
    return obstacles;
  }

  bool operator==(const Lattice & l2);
  //private:
  bool isObstacled(usint x, usint y, usint j, usint i) const;
  std::vector<bool> computeChromosomeLayout(const Population * pop, bool formale) const;
  void applyMutations(uint64_t * h1, uint64_t * h2, uint nummut, usint chrom_length, usint offset, int marker=-1) const;
  void applyRecombinations(uint64_t * h1, uint64_t * h2, uint numrek, usint chrom_length, usint offset) const;
  void selectChromosome(uint64_t * Fh1,uint64_t * Mh1, uint64_t * Mh2, usint chrom_length, usint offset, usint bitnum) const;
  mutable std::vector< std::pair<std::string, PopulationPtr> > pop_numbers;
  Tab tab_;

private:
  uint gennum;              // S   ==   ctor
  uint capacity_;           // -  n/a   ctor
  usint width,height;       // S   ==   ctor
  std::vector <Atab> atab;  // -    -   ctor
  Specimen ** positions1;
  usint max_father_dist;
  usint max_child_dist;
  std::vector <Line> obstacles; // S XX ctor
  Point * positions2;
  bool panmixia;
  bool promiscuous;
  Point *empty;
  uint empty_ind;
  bool altered;
  FemaleRandomizer F;
  std::auto_ptr<Environment> env;
  //std::vector<uint> rek_stat; //todo remove
  uint stats_period_;
};


BOOST_CLASS_VERSION(Lattice,2)

namespace boost
{
  namespace serialization
  {
    template<class Archive>
    inline void save_construct_data(Archive & ar, const Lattice * t, unsigned int /*file_version*/)
    {
      usint width=t->getWidth(); //dirty, lousy workaround of ambiguity with friend declaration. Sorry.
      usint height=t->getHeight();
      ar << width;
      ar << height;
    }

    template<class Archive>
    inline void load_construct_data(Archive & ar, Lattice * t, unsigned int /*file_version*/)
    {
      usint width,height;
      ar >> width;
      ar >> height;
      ::new(t)Lattice(width,height);
    }
  }
} // namespace ...

#endif /* LATTICE_HPP_ */
