#include "ext/tut.hpp"
#include <vector>
#include <boost/shared_ptr.hpp>
#include <memory>
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>
#include <boost/serialization/vector.hpp>
#include "detail/auto_ptr_serial.hpp"
#include "Lattice.hpp"
#include "detail/defs.hpp"

namespace
{

struct test_struct
{
  test_struct():L(64,64)
  {
    std::vector<uchar> v;
    v.push_back(4);
    v.push_back(4);
    P1.reset(new Population("Populacja",v,true));
    P2.reset(new Population("Populacja",v,false));
    P3.reset(new Population("Populacja3",v,false));
  }

  boost::shared_ptr<Population> P1,P2,P3;
  Lattice L;
};

typedef tut::test_group<test_struct> tf;
typedef tf::object object;
tf tests("Core/Lattice");
}

namespace tut
{

  //constructor
  template<>
  template<>
  void object::test<1>()
  {
  }

  //check isEmpty (lattice should be empty after creation)
  template<>
  template<>
  void object::test<2>()
  {
    for (uint i=0; i<L.getHeight(); i++)
      for (uint j=0; j<L.getWidth(); j++)
	ensure(L.isEmpty(j,i));
  }

  //width, height, gennum
  template<>
  template<>
  void object::test<3>()
  {
    ensure(L.getWidth()==64);
    ensure(L.getHeight()==64);
    ensure(L.getGennum()==0);
  }

  //fill
  template<>
  template<>
  void object::test<4>()
  {
    L.fill(P1);
  }

  //fill2
  template<>
  template<>
  void object::test<5>()
  {
    L.fill(P1);
    for (uint i=0; i<L.getHeight(); i++)
      for (uint j=0; j<L.getWidth(); j++)
	ensure(L.isEmpty(j,i)==false);
  }

  //fill3
  template<>
  template<>
  void object::test<6>()
  {
    L.fill(P1);
    for (uint i=0; i<L.getHeight(); i++)
      for (uint j=0; j<L.getWidth(); j++)
	ensure(L.isEmpty(j,i)==false);
    ensure(L.latticeCount()==64*64);
  }

  //fill4
  template<>
  template<>
  void object::test<7>()
  {
    ensure_equals("lattice count 0",L.latticeCount(),0U);
    ensure_equals("atab count 0",L.atabCount(),L.latticeCount());
    L.fill(P1,10,10,10,10);
    ensure_equals("lattice count 1",L.latticeCount(),100U);
    ensure_equals("atab count 1",L.atabCount(),L.latticeCount());
    L.fill(P1,20,20,10,10);
    ensure_equals("lattice count 2",L.latticeCount(),200U);
    ensure_equals("atab count 2",L.atabCount(),L.latticeCount());
    L.fill(P1,15,15,10,10);
    ensure_equals("lattice count 3",L.latticeCount(),250U);
    ensure_equals("atab count 3",L.atabCount(),L.latticeCount());
  }

  //kill
  template<>
  template<>
  void object::test<8>()
  {
    ensure_equals("lattice count 0",L.latticeCount(),0U);
    ensure_equals("atab count 0",L.atabCount(),L.latticeCount());
    L.fill(P1,10,10,10,10);
    ensure_equals("lattice count 1",L.latticeCount(),100U);
    ensure_equals("atab count 1",L.atabCount(),L.latticeCount());
    L.kill(4,4);
    ensure_equals("lattice count 2",L.latticeCount(),100U);
    ensure_equals("atab count 2",L.atabCount(),L.latticeCount());
    L.kill(15,4);
    ensure_equals("lattice count 3",L.latticeCount(),100U);
    ensure_equals("atab count 3",L.atabCount(),L.latticeCount());
    L.kill(14,12);
    ensure_equals("lattice count 4",L.latticeCount(),99U);
    ensure_equals("atab count 4",L.atabCount(),L.latticeCount());
  }

  //purge
  template<>
  template<>
  void object::test<10>()
  {
    ensure_equals("lattice count 0",L.latticeCount(),0U);
    ensure_equals("atab count 0",L.atabCount(),L.latticeCount());
    L.fill(P1,10,10,10,10);
    ensure_equals("lattice count 1",L.latticeCount(),100U);
    ensure_equals("atab count 1",L.atabCount(),L.latticeCount());
    L.purge(0.5);
    uint size=L.latticeCount();
    ensure(size>0.4999*100 && size<0.5001*100);
  }

  //two populations with same names
  template<>
  template<>
  void object::test<11>()
  {
    L.fill(P1,10,10,10,10);
    try
      {
	L.fill(P2,10,10,10,10);
	tut::fail("Registration of two populations with different names passed.");
      }
    catch(NchrException &e)
      {
	//expected;
      }
  }

  //two populations with different names
  template<>
  template<>
  void object::test<12>()
  {
    L.fill(P1,10,10,10,10);
    L.fill(P3,20,20,10,10);
  }

  //two populations with different names
  template<>
  template<>
  void object::test<13>()
  {
    L.fill(P1,10,10,10,10);
    L.fill(P3,20,20,10,10);
  }

  //getPopnum
  template<>
  template<>
  void object::test<14>()
  {
    L.fill(P1,1,1,1,1);
    L.fill(P3,2,2,1,1);
    ensure_equals("getpopnum 1",static_cast<int>(L.getPopNum(P1)),0);
    ensure_equals("getpopnum 2",static_cast<int>(L.getPopNum(P3)),1);
    //ensure_equals("getpopnum 3",static_cast<int>(L.getPopNum(P2)),6); //TODO poprawi rejestracj populacji
  }

  //copy
  template<>
  template<>
  void object::test<15>()
  {
    L.fill(P1,10,10,50,50);
    L.copy(L,10,10,20,20,40,0);

    for (uint i=10; i<30; i++)
      for (uint j=10; j<30; j++)
	ensure(L.getGuy(j,i)==L.getGuy(j+30,i-10));

    L.fill(P1,42,6,1,1);
    ensure(L.getGuy(12,16)!=L.getGuy(12+30,16-10));
  }

  //apply mutations, may fail spontaneously
  template<>
  template<>
  void object::test<16>()
  {
    uint64_t Mh1[4]={0U,0U,0U,0U};
    uint64_t Mh2[4]={0U,0U,0U,0U};

    L.applyMutations(Mh1,Mh2,5,2,1);

    ensure_equals("number of mutations",bit_count64(Mh1[1])+bit_count64(Mh1[2]),5);
    ensure_equals("number of mutations",bit_count64(Mh2[1])+bit_count64(Mh2[2]),5);

    ensure_equals("number of mutations out of bounds",bit_count64(Mh1[0])+bit_count64(Mh1[3]),0);
    ensure_equals("number of mutations out of bounds",bit_count64(Mh2[0])+bit_count64(Mh2[3]),0);
  }

  //apply recombinations
  template<>
  template<>
  void object::test<17>()
  {
    uint64_t Mh1[4]={0,0,0,0};
    uint64_t Mh2[4]={-1,-1,-1,-1};

    L.applyRecombinations(Mh1,Mh2,1,2,1);

    ensure_equals("recombinations out of bounds",Mh1[0],0U);
    ensure_equals("recombinations out of bounds",Mh2[0],uint64_t(-1));
    ensure_equals("recombinations out of bounds",Mh1[3],0U);
    ensure_equals("recombinations out of bounds",Mh2[3],uint64_t(-1));
    ensure_equals("recombination",Mh1[2]&1,1U);
    ensure_equals("recombination",Mh2[2]&1,0U);
  }

  //apply recombinations 2
  template<>
  template<>
  void object::test<18>()
  {
    uint64_t Mh1[4]={0,0,0,0};
    uint64_t Mh2[4]={-1,-1,-1,-1};

    L.applyRecombinations(Mh1,Mh2,11,2,1);

    ensure_equals("recombinations out of bounds",Mh1[0],0U);
    ensure_equals("recombinations out of bounds",Mh2[0],uint64_t(-1));
    ensure_equals("recombinations out of bounds",Mh1[3],0U);
    ensure_equals("recombinations out of bounds",Mh2[3],uint64_t(-1));
    ensure_equals("recombination",Mh1[2]&1,1U);
    ensure_equals("recombination",Mh2[2]&1,0U);
  }

//serialization
template<>
template<>
void object::test<19>()
{
  std::vector<uchar> v(2,2);
  std::stringstream s;
  std::auto_ptr<Lattice> La(new Lattice(128,128));
  std::auto_ptr<Lattice> Lb;

  {
    boost::archive::text_oarchive oa(s);
    oa << La;
  }

  {
    boost::archive::text_iarchive ia(s);
    ia >> Lb;
  }

  ensure((*La)==(*Lb));
}

}
