/*
 * PythonAPI.cpp
 *
 *  Created on: 2010-01-03
 *      Author: Wojciech Waga <wojciech.waga.com>
 */

#include <fstream>
#include <iostream>
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>
#include <boost/iostreams/filtering_stream.hpp>
#include <boost/iostreams/filter/bzip2.hpp>

#include "FileImg/Image.hpp"
#include "FileImg/ImageProperties.hpp"
#include "FileImg/ImgSnapshot.hpp"
#include "FileImg/ImgPopulSnapshot.hpp"
#include "FileImg/ImgChromosome.hpp"
#include "FileImg/ImgAge.hpp"
#include "FileImg/ImgLD.hpp"

#include "FileTxt/Text.hpp"
#include "FileTxt/TxtAge.hpp"
#include "FileTxt/TxtChromosome.hpp"
#include "FileTxt/TxtChrStat.hpp"
#include "FileTxt/PopCNT.hpp"
#include "FileODS/Content.hpp"

#include "LatticeAPI.hpp"

LatticeAPI::LatticeAPI(usint w, usint h)
{
  L.reset(new Lattice(w,h));
}

LatticeAPI::LatticeAPI(const LatticeAPI &p)
{
  throw NchrException("Cannot copy Population object!");
}

LatticeAPI::LatticeAPI(std::string filename)
{
  std::ifstream inf(filename.c_str(), std::ios_base::binary);
  if (!inf.good())
    throw NchrException("No such file or no read permision.");
  boost::iostreams::filtering_istream in;

  if (filename.rfind("nc2")==filename.size()-3) //bzipped dump
    in.push(boost::iostreams::bzip2_decompressor());
  else
    if (filename.rfind("np2")!=filename.size()-3) //plain dump
      throw NchrException("Unknown file type.");

  in.push(inf);
  boost::archive::text_iarchive ia(in);
  ia >> L;
  std::cout << "Loaded lattice with " << L->latticeCount()  << " individuals.\n";
}

LatticeAPI::~LatticeAPI()
{
  L.reset();
}

void LatticeAPI::save(const std::string &filename) const
{
  std::ofstream out((filename+".nc2").c_str(),std::ios_base::binary);
  boost::iostreams::filtering_ostream os2bz;
  os2bz.push(boost::iostreams::bzip2_compressor());
  os2bz.push(out);
  boost::archive::text_oarchive oa(os2bz);
  oa << L;
}
/*
void LatticeAPI::setInfertility(float p)
{
  L->setInfertility(p);
}
*/
void LatticeAPI::frozenGen()
{
  L->nexGen(true);
  if(sig_caught)
    {
      save("final_dump");
      throw NchrException("Exited due to a signal.");
    }

  std::ifstream inf("stop");
  if (inf.good())
    {
      std::cerr << "Finishing..." << std::endl;
      save("final_dump");
      throw NchrException("Exited due to a stop file.");
    }
}

void LatticeAPI::nexGen()
{
  L->nexGen();
  if(sig_caught)
    {
      save("final_dump");
      throw NchrException("Exited due to a signal.");
    }

  std::ifstream inf("stop");
  if (inf.good())
    {
      std::cerr << "Finishing..." << std::endl;
      save("final_dump");
      throw NchrException("Exited due to a stop file.");
    }
}

Specimen  LatticeAPI::getGuy(usint x, usint y) const
{
  return L->getGuy(x,y);
}

bool LatticeAPI::isEmpty(usint x, usint y) const
{
  return L->isEmpty(x,y);
}

usint LatticeAPI::getHeight() const
{
  return L->getHeight();
}

usint LatticeAPI::getWidth() const
{
  return L->getWidth();
}

void LatticeAPI::update()
{
  L->update();
}

void LatticeAPI::setEnvironment(Environment &e)
{
  L->setEnvironment(e);
}

void LatticeAPI::unsetEnvironment()
{
  L->unsetEnvironment();
}

void LatticeAPI::killSquare(usint x, usint y, usint w, usint h)
{
  L->killSquare(x,y,w,h);
}

void LatticeAPI::setImg(const std::string & key, int value)
{
  imgProp.setValue(key, value);
}
void LatticeAPI::setImg(const std::string & key, const std::string & value)
{
  imgProp.setValue(key, value);
}

void LatticeAPI::setImg(const std::string & key, const boost::python::list & list)
{
  imgProp.setValue(key, listToVec<int>(list));
}

void LatticeAPI::snpPngA(const std::string &filename) const
{
  snpPngB(filename,false);
}

void LatticeAPI::snpPngB(const std::string &filename, bool f) const
{
      ImgSnapshot img(imgProp,*L);
      img.write(filename, f);
}

void LatticeAPI::snpPop(const std::string &filename) const
{
      ImgPopulSnapshot img(imgProp,*L);
      img.write(filename);
}

void LatticeAPI::chrPngA(const std::string &filename) const
{
  chrPngC(filename, false);
}

void LatticeAPI::chrPngB(const std::string &filename, uint chr) const
{
  chrPngD(filename, chr, false);
}

void LatticeAPI::chrPngC(const std::string &filename, bool f) const
{
  ImgChromosome img(imgProp,*L);
  img.write(filename,f);
}

void LatticeAPI::chrPngD(const std::string &filename, uint chr, bool f) const
{
  ImgChromosome img(imgProp,*L);
  img.write(filename,chr,f);
}

void LatticeAPI::LDPng(const std::string &filename) const
{
  ImgLD img(imgProp,*L);
  img.write(filename);
}

void LatticeAPI::chrTxtA(const std::string &filename) const
{
  chrTxtC(filename, false);
}

void LatticeAPI::chrTxtB(const std::string &filename, uint chr) const
{
  chrTxtD(filename, chr, false);
}

void LatticeAPI::chrTxtC(const std::string &filename, bool f) const
{
  TxtChromosome txt(*L);
  txt.write(filename,f);
}

void LatticeAPI::chrTxtD(const std::string &filename, uint chr, bool f) const
{
  TxtChromosome txt(*L);
  txt.write(filename,chr,f);
}

void LatticeAPI::agePngA(const std::string &filename) const
{
  agePngC(filename, false);
}

void LatticeAPI::agePngB(const std::string &filename, uint pop) const
{
  agePngD(filename, pop, false);
}

void LatticeAPI::agePngC(const std::string &filename, bool f) const
{
  ImgAge img(imgProp,*L);
  img.write(filename,f);
}

void LatticeAPI::agePngD(const std::string &filename, uint pop, bool f) const
{
  ImgAge img(imgProp,*L);
  img.write(filename,pop,f);
}

void LatticeAPI::ageTxtA(const std::string &filename) const
{
  TxtAge txt(*L);
  txt.write(filename);
}

void LatticeAPI::ageTxtB(const std::string &filename, uint pop) const
{
  TxtAge txt(*L);
  txt.write(filename,pop);
}

void LatticeAPI::chrStatTxtA(const std::string &filename) const
{
  static bool first=true;

  if(first)
    {
      remove(filename.c_str());
      first=false;
    }

  TxtChrStat txt(*L);
  txt.write(filename);
}

void LatticeAPI::chrStatTxtB(const std::string &filename, uint pop) const
{
  static bool first=true;

  if(first)
    {
      remove(filename.c_str());
      first=false;
    }

  TxtChrStat txt(*L);
  txt.write(filename,pop);
}

void LatticeAPI::statistics(const std::string &filename) const
{
  char dirn[]="/tmp/NchrXXXXXX";
  const char * dirname=mkdtemp(dirn);
  //std::cerr << dirname;

  char path[1600];
  strncpy(path,dirname,60);
  strcat(path,"/META-INF");
  mkdir(path,0700);
  strcat(path,"/manifest.xml");
  {
    std::ofstream of(path);
    of << "<?xml version=\"1.0\" encoding=\"UTF-8\"?><manifest:manifest xmlns:man"
        "ifest=\"urn:oasis:names:tc:opendocument:xmlns:manifest:1.0\"><manifest:file-e"
        "ntry manifest:media-type=\"application/vnd.oasis.opendocument.spreadsheet\" "
        "manifest:version=\"1.2\" manifest:full-path=\"/\"/><manifest:file-entry manifest:m"
        "edia-type=\"text/xml\" manifest:full-path=\"content.xml\"/><manifest:file-entry m"
        "anifest:media-type=\"text/xml\" manifest:full-path=\"meta.xml\"/></manifest:manifest>";
  }

  {
    char mime[60];
    strncpy(mime,dirname,30);
    strcat(mime,"/mimetype");
    std::ofstream of(mime);
    of << "application/vnd.oasis.opendocument.spreadsheet";
  }

  {
    char mime[60];
    strncpy(mime,dirname,30);
    strcat(mime,"/meta.xml");
    std::ofstream of(mime);
    of << "<?xml version=\"1.0\" encoding=\"UTF-8\"?><office:document-meta xmlns:office"
        "=\"urn:oasis:names:tc:opendocument:xmlns:office:1.0\" xmlns:xlink=\"http://www.w3."
        "org/1999/xlink\" xmlns:dc=\"http://purl.org/dc/elements/1.1/\" xmlns:meta=\"urn:oasis"
        ":names:tc:opendocument:xmlns:meta:1.0\" xmlns:ooo=\"http://openoffice.org/2004/of"
        "fice\" office:version=\"1.2\"><office:meta><meta:creation-date>2010-09-24T17:35:33"
        "</meta:creation-date><dc:date>2010-09-24T18:15:56</dc:date><meta:editing-durat"
        "ion>PT00H10M05S</meta:editing-duration><meta:editing-cycles>2</meta:editing-cycl"
        "es><meta:generator>OpenOffice.org/3.1$Linux OpenOffice.org_project/310m19$Build"
        "-9420</meta:generator><meta:document-statistic meta:table-count=\"1\" meta:cell-co"
        "unt=\"100\" meta:object-count=\"0\"/></office:meta></office:document-meta>";
  }


  {
    std::string content(dirname);
    Content cnt(*L);
    cnt.write(content+"/content.xml");
  }


  char cwd[80];
  getcwd(cwd,80);
  sprintf(path,"cd %s; zip -q -r %s/%s.ods *",dirname,cwd,filename.c_str());
  system(path);
  sprintf(path,"rm -rf %s",dirname);
  system(path);
}

void LatticeAPI::popCntA(const std::string &filename) const
{
  popCntB(filename, false);
}

void LatticeAPI::popCntB(const std::string &filename, bool f) const
{
   PopCNT p(*L);
   p.write(filename, f);
}

void LatticeAPI::purge(float x)
{
  L->purge(x);
}

void LatticeAPI::fill1(PopulationAPI &p1)
{
  L->fill(p1.getP());
}

void LatticeAPI::fill2(PopulationAPI &p1, usint x, usint y, usint w, usint h)
{
  L->fill(p1.getP(),x,y,w,h);;
}

void LatticeAPI::setPanmixia(bool panm)
{
  L->setPanmixia(panm);
}

uint LatticeAPI::getGennum() const
{
  return L->getGennum();
}

void LatticeAPI::setStatsPeriod(uint gen)
{
  L->setStatsPeriod(gen);
}

void LatticeAPI::printStats(const std::string & pop, const std::string & filename) const
{
  std::ofstream out(filename.c_str());

  std::vector<uint> v=L->getStats(pop);
  for (std::vector<uint>::const_iterator it=v.begin(); it!=v.end(); ++it)
    out << *it << std::endl;
}

boost::python::list LatticeAPI::listPopulations() const
{
  boost::python::list t;
  std::vector< std::pair<std::string, Lattice::PopulationPtr> > tmp=L->listPopulations();
  for(std::vector< std::pair<std::string, Lattice::PopulationPtr> >::const_iterator it=tmp.begin(); it!=tmp.end(); ++it)
              t.append(it->first);
  return t;
}

PopulationAPI  LatticeAPI::getPopulation(const std::string & s) const
{
  return PopulationAPI(L->getPopulation(s));
}

void LatticeAPI::copy(const LatticeAPI & L2, usint src_x, usint src_y, usint w, usint h, usint dst_x, usint dst_y)
{
  L->copy(*(L2.L),src_x,src_y,w,h,dst_x,dst_y);
}

bool LatticeAPI::operator==(const LatticeAPI & l) const
{
  return (*L==*l.L);
}
