/*
 * LatticeAPI.hpp
 *
 *  Created on: 2010-01-03
 *      Author: Wojciech Waga <wojciech.waga.com>
 */

#ifndef LATTICEAPI_HPP_
#define LATTICEAPI_HPP_

#include <string>
#include <vector>
#include <boost/python.hpp>
#include <boost/python/docstring_options.hpp>

#include "detail/defs.hpp"
#include "detail/NchrException.hpp"
#include "Population.hpp"
#include "Lattice.hpp"

#include "FileImg/ImageProperties.hpp"

#include "PopulationAPI.hpp"

/**
 * @brief Exports functionality of the Lattice class to a python interface.
 * For more comments please refer to the Lattice class.
 */
class LatticeAPI
{
public:
  /**
   * @brief creates a Lattice object.
   * @param w Width of the lattice to create.
   * @param h Height of the lattice to create
   */
  LatticeAPI(usint w, usint h);
  LatticeAPI(const LatticeAPI &p);

  /**
   * @brief Loads a lattice from a file.
   * @param filename Name of the file to load.
   */
  LatticeAPI(const std::string filename);
  ~LatticeAPI();

  void nexGen();
  void frozenGen();
  uint getGennum() const;
  void update();
  void setEnvironment(Environment &e);
  void unsetEnvironment();
  void killSquare(usint x, usint y, usint w, usint h);
  void setPanmixia(bool p);

  void fill1(PopulationAPI &p1);
  void fill2(PopulationAPI &p1, usint x, usint y, usint w, usint h);

  void setImg(const std::string & key, int value);
  void setImg(const std::string & key, const std::string & value);
  void setImg(const std::string & key, const boost::python::list & list);

  void snpPngA(const std::string &filename) const;
  void snpPngB(const std::string &filename, bool f) const;

  void snpPop(const std::string &filename) const;

  void chrPngA(const std::string &filename) const;
  void chrPngB(const std::string &filename, uint chr) const;
  void chrPngC(const std::string &filename, bool f) const;
  void chrPngD(const std::string &filename, uint chr, bool f) const;

  void chrTxtA(const std::string &filename) const;
  void chrTxtB(const std::string &filename, uint chr) const;
  void chrTxtC(const std::string &filename, bool f) const;
  void chrTxtD(const std::string &filename, uint chr, bool f) const;

  void LDPng(const std::string &filename) const;

  void agePngA(const std::string &filename) const;
  void agePngB(const std::string &filename, uint pop) const;
  void agePngC(const std::string &filename, bool f) const;
  void agePngD(const std::string &filename, uint pop, bool f) const;

  void ageTxtA(const std::string &filename) const;
  void ageTxtB(const std::string &filename, uint pop) const;

  /**
   * @brief Appends chromosomes statistics to a file
   * First run deletes the file, each next run appends to the existing file.
   * @param filename filename to use
   */
  void chrStatTxtA(const std::string &filename) const;
  void chrStatTxtB(const std::string &filename, uint pop) const;

  /**
   * @brief Generates ODS file with statistics for the simulation
   * @param filename file name to write stats
   */
  void statistics(const std::string &filename) const;

  void popCntA(const std::string &filename) const;
  void popCntB(const std::string &filename, bool f) const;
  void purge(float x);

  void setStatsPeriod(uint b);
  void printStats(const std::string & pop, const std::string & filename) const;

  void copy(const LatticeAPI & L2, usint src_x, usint src_y, usint w, usint h, usint dst_x, usint dst_y);

  boost::python::list listPopulations() const;
  PopulationAPI getPopulation(const std::string & s) const;

  Specimen getGuy(usint x, usint y) const;
  bool isEmpty(usint x, usint y) const;

  usint getHeight() const;
  usint getWidth() const;

  void save(const std::string &filename) const;

  bool operator==(const LatticeAPI & l) const;

private:
  /**
   * @brief Helper function to create vectors of parameters from python lists.
   * @param list Python-style list of arguments.
   * @return Vector containing all the elements from input python list.
   */
  template <class T>
  const std::vector<T> listToVec(const boost::python::list &list) const
  {
    const int n = boost::python::extract<int>(list.attr("__len__")());
    std::vector<T> chr;
    for ( int i = 0; i < n; i++ )
      chr.push_back (boost::python::extract<T>((list)[i]));
    return chr;
  }

private:
  std::auto_ptr<Lattice> L;
  ImageProperties imgProp;
};

#endif /* LATTICEAPI_HPP_ */

