/*
 * Population.hpp
 *
 *  Created on: 2009-12-30
 *      Author: Wojciech Waga <wojciech.waga.com>
 */

#ifndef POPULATION_HPP_
#define POPULATION_HPP_

#include <string>
#include <vector>
#include <boost/noncopyable.hpp>
#include <boost/serialization/access.hpp>
#include <boost/serialization/version.hpp>

#include "detail/defs.hpp"
#include "detail/NchrException.hpp"
#include "detail/Poisson.hpp"
#include "detail/RecognitionMarkers.hpp"

class Population : public boost::noncopyable, private boost::equality_comparable<Population>
{
public:
  /**
   * @brief A constructor. Creates a new population object with initial parameters.
   * Populations on the same lattice should all have different names. Number of
   * chromosomes and all other parameters can be either different or not.
   * @param n name of the population, should be distinct within given lattice
   * @param vec vector of chromosome lengths
   * @param hasY whether or not there are sex chromosomes
   */
  Population(std::string n, const std::vector<uchar> & vec, bool hasY);

  /**
   * @brief Sets number of defects that kills individual.
   * @param T number of defects
   */
  void setT(usint T){altered=true; t=T;}

  /**
   * @brief Sets number of conceptions to perform at each generation.
   * @param B number to set
   */
  void setB(float B){altered=true; b=B; B_.setLambda(B);}

  /**
   * @brief Set distance for partner lookup.
   * @param F radius to set
   */
  void setFatherDist(usint F) {altered=true; f=F;}

  /**
   * @brief Set distance for offspring placement.
   * @param C radius to set
   */
  void setChildDist(usint C) {altered=true; c=C;}

  /**
   * @brief Set life span of individuals.
   * Life span is defined as a maximum age that healthy individual can reach. It is usually set to 64, 128 or 256.
   * Lenghts of all chromosomes should be divisible by the life span, in particular they can't be shorter.
   * @param ls life span to set
   */
  void setLifeSpan(usint ls) {altered=true; life_span=ls;}

  /**
   * @brief Set birth age of individuals.
   * Birth age is an 'age' with which individuals are born. It is meant to mimic prenatal period and gene
   * expression before birth.
   * @param b birth age to set
   */
  void setBirthAge(usint b) {altered=true; ba=b;}

  /**
   * @brief Set reproduction age
   * Reproduction age is the minimal age at which individuals can mate. There is no upper limit. This parameter
   * is cruicial for life expectancy.
   * @param r age to set
   */
  void setReproductionAge(usint r) {altered=true; ra=r;}

  /**
   * @brief Set whether the population can use FIVET
   */
  void useFivet(bool b)
  {
    has_fivet_ = b;
  }

  /**
   * @brief return whether or not we can use FIVET
   * @return true if we can use FIVET
   */
  bool hasFivet() const
  {
    return has_fivet_;
  }

  /**
   * @brief Set factor of mutations for individuals after ART.
   * This number is a factor that defines the amount of mutations after ART. I.e. if this
   * number is 2.5 individuals conveived with ART will have 2.5x as much mutations.
   * @param f factor to set
   */
  void setFivetFactor(float f)
  {
    if (f<0)
      throw NchrException("Fivet factor should be nonzero.");
    fivet_factor=f;
    mutations_increased_.setLambda(mutation_rate_list,fivet_factor*mutation_factor);
  }

  /**
   * @brief Set fraction of infertile individuals
   * @param f - fraction (0..1)
   */
  void setInfertility(float f)
  {
    infertility_=f;
  }

  /**
   * @brief Get fraction of infertile individuals
   */
  float getInfertility() const
  {
    return infertility_;
  }

  /**
   * @brief Set fraction of infertile individuals born with FIVET
   * @param f - fraction (0..1)
   */
  void setInfertilityAF(float f)
  {
    infertilityAF_=f;
  }

  /**
   * @brief Get fraction of infertile individuals born with FIVET
   */
  float getInfertilityAF() const
  {
    return infertilityAF_;
  }

  /**
   * @brief Set fraction of cases where FIVET gives birth to an offspring
   * @param f - fraction (0..1)
   */
  void setFIVETefficacy(float f)
  {
    FIVETefficacy_=f;
  }

  /**
   * @brief Get fraction of cases where FIVET gives birth to an offspring
   */
  float getFIVETefficacy() const
  {
    return FIVETefficacy_;
  }

  /**
   * @brief Set mutation rate
   * Mutation rate is separate for each chromosome. Then mutation rate set to 1 defines
   * num_chromosomes mutations introduced to genome in each generation.
   * @param M number of mutations per chromosome per generation
   */
  void setMutationRate(float M);

  /**
   * @brief Set mutation rate
   * Defines a vector of mutation rates for all chromosomes. Length of this vector should
   * be equal to the number of chromosomes in genome.
   * @param val vector of mutation rates to set
   */
  void setMutationRate(const fvector &val);

  /**
   * @brief Get vector of mutation rates.
   */
  const fvector & getMutationRate() const
  {
    return mutation_rate_list;
  }

  /**
   * @brief Set mutation factor.
   * Mutation factor is the number by which all mutation rates are multiplied. It is
   * used to easily increase mutation rates without changing too much code.
   * @param factor by which increase mutation rates
   */
  void setMutationFactor(float f);

  /**
   * @brief Set recombination factor.
   * Recombination factor is the number by which all recombination rates are multiplied.
   * It is used to easily increase recombination rates without changing too much code.
   * @param factor by which increase recombination rates
   */
  void setRecombinationFactor(float r);

  /**
   * @brief Set the same recombination rate for all chromosomes.
   * @param R recombination rate to set
   */
  void setRecombRate(float R);

  /**
   * @brief Set vector of recombination rates.
   * Sets different recombination rate for each pair of chromosomes. Length of this
   * vector should be equal to the number of chromosomes.
   * @param val vector of recombination rates
   */
  void setRecombRate(const fvector &val);

  /**
   * @brief Get recombination rates.
   * Gets vector of recombination rates for all chromosomes.
   */
  const fvector & getRecombRate() const
  {
    return recombination_rate_list;
  }

  /**
   * @brief Sets markers used for chromosome recognition
   * @param val - vector of numbers of bits to set market at
   */
  void setReconMarker(usint chr, usint bit);

  char getInitType() const
  {
    return initType;
  }

  uint getInitVal() const
  {
    return initVal;
  }

  usint getLifeSpan() const { return life_span;}

  /**
   * @brief Get number of chromosomes in the genome.
   */
  usint getNumChromosomes() const {return chromosomes.size();}

  /**
   * @brief Get number of chunks in the genome.
   * Chunks are of the length 64 nucleotides. And every chromosome should
   * have length divisible by 64.
   */
  usint getNumChunks() const {return num_chunks;}

  usint getT() const {return t;}
  float getB() const {return b;}
  float getBPoisson() const {return B_.getNum(0);}
  usint getFatherDist() const {return f;}
  usint getChildDist() const {return c;}
  usint getBirthAge() const {return ba;}
  usint getReproductionAge() const {return ra;}

  /**
   * @brief Check whether or not population has sex chromosomes.
   * @return true if the last pair of chromosomes contains X and Y
   */

  bool hasYchr() const {return Y;}

  /**
   * @brief Get lengths of all chromosomes in chunks.
   */
  const std::vector<uchar> & getChromLengths() const {return chromosomes;}

  /**
   * @brief Get length of a particular chromosome.
   @ @param chr number of chromosome
  */
  usint getChromLength(usint chr) const {return chromosomes.at(chr);}

  /**
   * @brief Get offset of a particular chromosome.
   * Offset is the number of chunks in all preceeding chromosomes.
   * @param chr number of chromosome
   */
  usint getChromOffset(usint chr) const {return chrom_offsets.at(chr);}

  /**
   * @brief Set initialization type
   * Initialization type specifies what will genomes of new individuals look like.
   * They can be with some random mutations or clean.
   */
  void init(std::string s);

  /**
   * @brief Get name of this population.
   * Name cannot be changed and should be unique within the lattice.
   */
  const std::string& getName() const
  {
    return name;
  }

  /**
   * @brief Check if genomes of two population are of the same type.
   * Genomes are of the same type when there is the same number
   * of chromosomes and corresponding chromosomes have the same lengths
   * in both populations.
   */
  bool sameLayout(const Population *p) const
  {
    if (p->getNumChunks()!=getNumChunks())
      return false;

    if (p->getNumChromosomes()!=getNumChromosomes())
      return false;

    for (usint i=0; i<getNumChromosomes(); i++)
      if (p->chromosomes.at(i)!=chromosomes.at(i))
        return false;

    if (p->hasYchr()!=hasYchr())
      return false;

    return true;
  }

  void read() const
  {
    altered=false;
  }

  void setPromisc(bool p)
  {
    promiscuous=p;
  }

  bool isPromisc()
  {
    return promiscuous;
  }

  bool isAltered() const
  {
    return altered;
  }

  /**
   * @brief Check if there is recognition on a given chromosome
   * @param chr chromosome number to check the recognition
   */

  bool hasRecognition(usint chr) const
  {
    return recognition_markers.isSet(chr);
  }

  /**
   * @brief Get position of a marker on a given chromosome
   * @param chr Chromosome number to get the marker
   */
  uchar getMarkerBit(usint chr) const
  {
    return recognition_markers.getMarker(chr);
  }

  bool hasReconMarkers() const
  {
    return !recognition_markers.isEmpty();
  }

  friend class boost::serialization::access;

  /**
   * @brief Serialization for saving and loading files.
   */
  template<class Archive>
  void serialize(Archive & ar, const unsigned int version)
  {
    ar & name;
    ar & t;
    ar & b;
    ar & f;
    ar & c;
    ar & life_span;
    ar & recombination_rate_list;
    ar & mutation_rate_list;
    ar & Y;
    ar & panmixia;
    ar & initType;
    ar & initVal;
    ar & num_chunks;
    ar & ba;
    ar & ra;
    ar & mutation_factor;
    ar & fivet_factor;
    ar & has_fivet_;
    ar & recognition_markers;
    ar & infertility_;
    ar & infertilityAF_;
    ar & FIVETefficacy_;
    boost::serialization::split_member(ar, *this, version);
  }

  template<class Archive>
  void load(Archive & ar, const unsigned int version)
  {
    if(version>=5)
      {
        ar & promiscuous;
        ar & recombination_factor;
      }

    altered=true;

    B_.setLambda(b);

    if(recombination_rate_list.size()!=0)
      recombinations_.setLambda(recombination_rate_list,recombination_factor);

    if(mutation_rate_list.size()!=0)
      {
        mutations_.setLambda(mutation_rate_list,mutation_factor);
        mutations_increased_.setLambda(mutation_rate_list,fivet_factor*mutation_factor);
      }
  }

  template<class Archive>
  void save(Archive & ar, const unsigned int version) const
  {
    ar & promiscuous;
    ar & recombination_factor;
  }

  bool operator==(const Population& p2) const
  {
    if (name!=p2.name) return false;
    if (t!=p2.t) return false;
    if (!compareDouble(b,p2.b) || f!=p2.f || c!=p2.c || life_span!= p2.life_span) return false;
    if (chromosomes!=p2.chromosomes) return false;
    if (chrom_offsets!=chrom_offsets) return false;
    if (!compareFloatVec(recombination_rate_list,p2.recombination_rate_list)) return false;
    if (!compareFloatVec(mutation_rate_list,p2.mutation_rate_list)) return false;
    if (!compareDouble(mutation_factor,p2.mutation_factor)) return false;
    if (!compareDouble(recombination_factor,p2.recombination_factor)) return false;
    if (!compareDouble(fivet_factor,p2.fivet_factor)) return false;
    if (Y!=p2.Y || panmixia!=p2.panmixia || has_fivet_!=p2.has_fivet_) return false;
    if (initVal!=p2.initVal || num_chunks!=p2.num_chunks) return false;
    if (ba!=p2.ba || ra!=p2.ra) return false;
    if (recognition_markers!=p2.recognition_markers) return false;
    if (initType!=p2.initType) return false;
    if (recombinations_!=p2.recombinations_) return false;
    if (mutations_!=p2.mutations_) return false;
    if (mutations_increased_!=p2.mutations_increased_) return false;
    if (B_!=p2.B_) return false;
    if (!compareDouble(infertility_,p2.infertility_) || !compareDouble(infertilityAF_,p2.infertilityAF_)) return false;
    if (!compareDouble(FIVETefficacy_,p2.FIVETefficacy_)) return false;
    return true;
  }

private:
  std::string name;
  usint t;
  float b;
  usint f,c,life_span;
  std::vector<uchar> chromosomes;
  std::vector<usint> chrom_offsets;
  fvector recombination_rate_list;
  fvector mutation_rate_list;
  float mutation_factor, fivet_factor, recombination_factor;
  bool Y,panmixia,has_fivet_,promiscuous;
  uint initVal;
  uint num_chunks;
  usint ba,ra;
  RecognitionMarkers recognition_markers;
  mutable bool altered;
  char initType;
public:
  Poisson recombinations_;
  Poisson mutations_;
  Poisson mutations_increased_;
  Poisson B_;
  float infertility_, infertilityAF_, FIVETefficacy_;
};

BOOST_CLASS_VERSION(Population, 5)
namespace boost
{
  namespace serialization
  {
    template<class Archive>
    inline void save_construct_data(Archive & ar, const Population * t, const unsigned int file_version)
    {
      // save data required to construct instance
      std::string s=t->getName(); //dirty, lousy workaround of ambiguity with friend declaration. Sorry.
      bool Y=t->hasYchr();
      std::vector<uchar> chromosomes=t->getChromLengths();
      ar << s;
      ar << Y;
      ar << chromosomes;
    }

    template<class Archive>
    inline void load_construct_data(Archive & ar, Population * t, const unsigned int file_version)
    {
      // retrieve data from archive required to construct new instance
      std::string s;
      bool Y;
      std::vector<uchar> chromosomes;

      ar >> s;
      ar >> Y;
      ar >> chromosomes;

      // invoke inplace constructor to initialize instance of my_class
      ::new(t)Population(s,chromosomes,Y);
    }
  }
} // namespace ...

#endif /* POPULATION_HPP_ */
