/*
 * Specimen.hpp
 *
 *  Created on: 2009-12-30
 *      Author: Wojciech Waga <wojciech.waga.com>
 */

#ifndef SPECIMEN_HPP_
#define SPECIMEN_HPP_

#include <stdint.h>
#include <boost/serialization/access.hpp>
#include <boost/serialization/version.hpp>
#include <boost/operators.hpp>
#include <iostream>
#include <stdexcept>

#include "detail/defs.hpp"
#include "detail/NchrException.hpp"
#include "Population.hpp"

class Specimen : private boost::equality_comparable<Specimen>
{
private:

public:
  Specimen();
  Specimen(usint nc, bool rl=false);
  ~Specimen();
  Specimen(const Specimen & ref);
  Specimen& operator=(const Specimen& b); //required by boost::serialization
  bool operator==(const Specimen& b) const;

  void realloc(usint n);

  void randomHaplotypes(usint n);
  void zeroHaplotypes();
  void testHaplotypes();

  void setAge(usint a)
  {
    age=a;
  }

  void setGender(Gender g)
  {
    gender=g;
  }

  Gender getGender() const
  {
    return gender;
  }

  void setPop(uchar p)
  {
    population=p;
  }

  uchar getPop() const
  {
    return population;
  }

  usint getAge() const
  {
    return age;
  }

  void incAge()
  {
    age++;
  }

  bool getY() const
  {
    return which_Y;
  }

  void setY(bool y)
  {
    which_Y=y;
  }

  usint getNumChunks() const
  {
    return num_chunks;
  }

  void setNumChunks(usint n)
  {
    num_chunks=n;
  }

  void setMarker(usint pos)
  {
    const uint cx=63-pos%64;
    const uint cxc=pos/64;
    h1[cxc]&=~(0x1ULL<<cx);
    h2[cxc]|=(0x1ULL<<cx);
  }

  void setInfertile(bool i)
  {
    infertile_=i;
  }

  bool isInfertile() const
  {
    return infertile_;
  }

  void setFIVETed(bool f)
  {
    FIVETed=f;
  }

  bool isFIVETed() const
  {
    return FIVETed;
  }

  friend class boost::serialization::access;

  template<class Archive>
  void serialize(Archive & ar, const unsigned int version)
  {
    uint death_year=0;
    ar & age;
    ar & death_year; //unused, to be removed
    ar & gender;
    ar & which_Y;
    ar & population;
    ar & num_chunks;

    if (version>=3)
      ar & FIVETed;

    if (num_chunks)
      for (uint i=0; i<num_chunks; i++)
	{
          ar & h1[i];
          ar & h2[i];
	}
    else
      {
        h1=0;
        h2=0;
      }
    ar & infertile_;
  }

  uint64_t getH1(usint n) const
  {
    if (n>=num_chunks)
      throw NchrException("Invalid chunk number");
    return h1[n];
  }

  uint64_t getH2(usint n) const
  {
    if (n>=num_chunks)
      throw NchrException("Invalid chunk number");
    return h2[n];
  }

  /**
   * @brief Initialize individual according to parameters of this population.
   * Age and gender are set randomly.
   */
  void initialize(const Population &p);

  /**
   * @brief Initialize individual according to parameters of this population.
   * Age is set randomly.
   * @param g gender to set.
   */
  void initialize(const Population &p, Gender g);


  //bool Atab::isLoser(const Population & pop, usint x, usint y, usint d_year, const Environment * env) const;

private:
  usint num_chunks;
public:
  uint64_t *h1, *h2;
private:
  usint age;
  bool which_Y;
  uchar population;
  Gender gender;
  bool infertile_;
  bool FIVETed;
};

BOOST_CLASS_VERSION(Specimen, 3)

namespace boost
{
  namespace serialization
  {
    template<class Archive>
    inline void save_construct_data(Archive & ar, const Specimen * t, const unsigned int file_version)
    {
      // save data required to construct instance
      usint nc=t->getNumChunks();
      ar << nc;
    }

    template<class Archive>
    inline void load_construct_data(Archive & ar, Specimen * t, const unsigned int file_version)
    {
      // retrieve data from archive required to construct new instance
      usint nc;
      ar >> nc;
      // invoke inplace constructor to initialize instance of Specimen
      ::new(t)Specimen(nc,true);
    }
  }
} // namespace ...

typedef std::vector<Specimen> vectSp;
typedef std::vector<vectSp> vectSpSp;
typedef std::vector<Specimen>::iterator vectSpIt;
typedef std::vector<vectSp>::iterator vectSpSpIt;


#endif /* SPECIMEN_HPP_ */

