/*
 * Atab.cpp
 *
 *  Created on: 2009-12-31
 *      Author: Wojciech Waga <wojciech.waga.com>
 */

#include<assert.h>
#include "Atab.hpp"
#include "../Lattice.hpp"

Atab::Atab(boost::shared_ptr<Population> p):specimen_count(0),F_specimen_count(0),pop(p),gennum(0),iterators_invalidated(true),pop_stats(p)
{
  aging_tab=new Point**[p->getLifeSpan()];
  atab_ind=new uint*[p->getLifeSpan()];
  max_atab=new uint*[p->getLifeSpan()];

  for (usint i=0; i<p->getLifeSpan(); i++)
    {
      aging_tab[i]=new Point*[2];
      atab_ind[i]=new uint[6];
      max_atab[i]=new uint[2];
    }

  for (usint i=0; i<p->getLifeSpan(); i++)
    {
      aging_tab[i][MALE]=0;
      aging_tab[i][FEMALE]=0;
      atab_ind[i][MALE]=0;
      atab_ind[i][FEMALE]=0;
      atab_ind[i][INF_MALE]=0;  //infertile males
      atab_ind[i][INF_FEMALE]=0;//infertile females (they both are included in the previous counters)
      atab_ind[i][FIVETed_MALE]=0;   //males born after IVF
      atab_ind[i][FIVETed_FEMALE]=0; //females born after IVF (both also included in the previous ones)
      max_atab[i][MALE]=0;
      max_atab[i][FEMALE]=0;
    }
  males_in_reproduction_period.resize(p->getLifeSpan());
}

Atab::Atab(const Atab & ref):specimen_count(ref.specimen_count),F_specimen_count(ref.F_specimen_count),pop(ref.pop),gennum(ref.gennum),iterators_invalidated(true),pop_stats(ref.pop_stats)
{
  aging_tab=new Point**[pop->getLifeSpan()];
  atab_ind=new uint*[pop->getLifeSpan()];
  max_atab=new uint*[pop->getLifeSpan()];

  for (usint i=0; i<pop->getLifeSpan(); i++)
    {
      aging_tab[i]=new Point*[2];
      atab_ind[i]=new uint[6];
      max_atab[i]=new uint[2];
    }

  for (usint i=0; i<pop->getLifeSpan(); i++)
    {
      atab_ind[i][MALE]=ref.atab_ind[i][MALE];
      atab_ind[i][FEMALE]=ref.atab_ind[i][FEMALE];
      atab_ind[i][INF_MALE]=ref.atab_ind[i][INF_MALE];
      atab_ind[i][INF_FEMALE]=ref.atab_ind[i][INF_FEMALE];
      atab_ind[i][FIVETed_MALE]=ref.atab_ind[i][FIVETed_MALE];
      atab_ind[i][FIVETed_FEMALE]=ref.atab_ind[i][FIVETed_FEMALE];
      max_atab[i][MALE]=ref.max_atab[i][MALE];
      max_atab[i][FEMALE]=ref.max_atab[i][FEMALE];

      if (max_atab[i][MALE])
        {
	  aging_tab[i][MALE]=new Point[max_atab[i][MALE]];
	  for (uint k=0; k<atab_ind[i][MALE]; k++)
	    aging_tab[i][MALE][k]=ref.aging_tab[i][MALE][k];
        } else
          aging_tab[i][MALE]=0;

      if (max_atab[i][FEMALE])
        {
	  aging_tab[i][FEMALE]=new Point[max_atab[i][FEMALE]];
	  for (uint k=0; k<atab_ind[i][FEMALE]; k++)
	    aging_tab[i][FEMALE][k]=ref.aging_tab[i][FEMALE][k];
        } else
          aging_tab[i][FEMALE]=0;
    }
  males_in_reproduction_period=ref.males_in_reproduction_period;
}

Atab & Atab::operator=(const Atab & ref)
{
  for (usint i=0; i<pop->getLifeSpan(); i++)
    {
      if (aging_tab[i][MALE]) delete [] aging_tab[i][MALE];
      if (aging_tab[i][FEMALE]) delete [] aging_tab[i][FEMALE];
    }

  for (usint i=0; i<pop->getLifeSpan(); i++)
    {
      delete[] aging_tab[i];
      delete[] atab_ind[i];
      delete[] max_atab[i];
    }

  delete[] aging_tab;
  delete[] atab_ind;
  delete[] max_atab;

  specimen_count=ref.specimen_count;
  F_specimen_count=ref.F_specimen_count;
  pop=ref.pop;
  gennum=ref.gennum;

  aging_tab=new Point**[pop->getLifeSpan()];
  atab_ind=new uint*[pop->getLifeSpan()];
  max_atab=new uint*[pop->getLifeSpan()];

  for (usint i=0; i<pop->getLifeSpan(); i++)
    {
      aging_tab[i]=new Point*[2];
      atab_ind[i]=new uint[6];
      max_atab[i]=new uint[2];
    }

  for (usint i=0; i<pop->getLifeSpan(); i++)
    {
      atab_ind[i][MALE]=ref.atab_ind[i][MALE];
      atab_ind[i][FEMALE]=ref.atab_ind[i][FEMALE];
      atab_ind[i][INF_MALE]=ref.atab_ind[i][INF_MALE];
      atab_ind[i][INF_FEMALE]=ref.atab_ind[i][INF_FEMALE];
      atab_ind[i][FIVETed_MALE]=ref.atab_ind[i][FIVETed_MALE];
      atab_ind[i][FIVETed_FEMALE]=ref.atab_ind[i][FIVETed_FEMALE];
      max_atab[i][MALE]=ref.max_atab[i][MALE];
      max_atab[i][FEMALE]=ref.max_atab[i][FEMALE];

      if (max_atab[i][MALE])
        {
          aging_tab[i][MALE]=new Point[max_atab[i][MALE]];
          for (uint k=0; k<atab_ind[i][MALE]; k++)
            aging_tab[i][MALE][k]=ref.aging_tab[i][MALE][k];
        }

      if (max_atab[i][FEMALE])
        {
          aging_tab[i][FEMALE]=new Point[max_atab[i][FEMALE]];
          for (uint k=0; k<atab_ind[i][FEMALE]; k++)
            aging_tab[i][FEMALE][k]=ref.aging_tab[i][FEMALE][k];
        }
    }
  males_in_reproduction_period=ref.males_in_reproduction_period;
  return *this;
}

Atab::~Atab()
{
  for (usint i=0; i<pop->getLifeSpan(); i++)
    {
      if (aging_tab[i][MALE]) delete [] aging_tab[i][MALE];
      if (aging_tab[i][FEMALE]) delete [] aging_tab[i][FEMALE];
    }

  for (usint i=0; i<pop->getLifeSpan(); i++)
    {
      delete[] aging_tab[i];
      delete[] atab_ind[i];
      delete[] max_atab[i];
    }

  delete[] aging_tab;
  delete[] atab_ind;
  delete[] max_atab;
}

void Atab::scroll(Tab &tab, Lattice & L)
{
  iterators_invalidated=true;
  //  std::cerr << "scroll" << std::endl;
  //kill individuals at the age of life_span
  for (uint i=0; i<atab_ind[pop->getLifeSpan()-1][MALE]; i++) //iteration over an entire row of aging tab for males
    {
      const usint posx=(aging_tab[pop->getLifeSpan()-1][MALE])[i].first;
      const usint posy=(aging_tab[pop->getLifeSpan()-1][MALE])[i].second;
      if (tab.get(posx, posy).isFIVETed()) F_specimen_count--;
      tab.setAlive(posx,posy,false);
      specimen_count--;
      if(L.isPanmictic())
        L.panPutFreePlace(posx,posy);
    }

  for (uint i=0; i<atab_ind[pop->getLifeSpan()-1][FEMALE]; i++)//the same but for females
    {
      const usint posx=(aging_tab[pop->getLifeSpan()-1][FEMALE])[i].first;
      const usint posy=(aging_tab[pop->getLifeSpan()-1][FEMALE])[i].second;
      if (tab.get(posx, posy).isFIVETed()) F_specimen_count--;
      tab.setAlive(posx,posy,false);
      specimen_count--;
      if(L.isPanmictic())
        L.panPutFreePlace(posx,posy);
    }

  delete [] aging_tab[pop->getLifeSpan()-1][MALE];
  delete [] aging_tab[pop->getLifeSpan()-1][FEMALE];

  aging_tab[pop->getLifeSpan()-1][MALE]=0;
  aging_tab[pop->getLifeSpan()-1][FEMALE]=0;

  //scroll the atab and atab_ind

  for (usint i=pop->getLifeSpan()-1; i!=0; i--)
    {
      aging_tab[i][MALE]=aging_tab[i-1][MALE];
      aging_tab[i][FEMALE]=aging_tab[i-1][FEMALE];

      atab_ind[i][MALE]=atab_ind[i-1][MALE];
      atab_ind[i][FEMALE]=atab_ind[i-1][FEMALE];

      atab_ind[i][INF_MALE]=atab_ind[i-1][INF_MALE];
      atab_ind[i][INF_FEMALE]=atab_ind[i-1][INF_FEMALE];

      atab_ind[i][FIVETed_MALE]=atab_ind[i-1][FIVETed_MALE];
      atab_ind[i][FIVETed_FEMALE]=atab_ind[i-1][FIVETed_FEMALE];

      max_atab[i][MALE]=max_atab[i-1][MALE];
      max_atab[i][FEMALE]=max_atab[i-1][FEMALE];

      for (uint j=0; j<atab_ind[i][MALE]; j++)
	{
	  const usint posx=aging_tab[i][MALE][j].first;
	  const usint posy=aging_tab[i][MALE][j].second;
	  tab.get(posx,posy).incAge();
	}

      for (uint j=0; j<atab_ind[i][FEMALE]; j++)
	{
	  const usint posx=aging_tab[i][FEMALE][j].first;
	  const usint posy=aging_tab[i][FEMALE][j].second;
	  tab.get(posx,posy).incAge();
	}
    }

  atab_ind[0][MALE]=0;
  atab_ind[0][FEMALE]=0;

  atab_ind[0][INF_MALE]=0;
  atab_ind[0][INF_FEMALE]=0;

  atab_ind[0][FIVETed_MALE]=0;
  atab_ind[0][FIVETed_FEMALE]=0;

  max_atab[0][MALE]=1000;
  max_atab[0][FEMALE]=1000;

  aging_tab[0][MALE]=new Point [max_atab[0][MALE]];
  aging_tab[0][FEMALE]=new Point [max_atab[0][FEMALE]];

  //genetic death

  const Environment * env=L.getEnvironment();

  for (usint i=1; i<pop->getLifeSpan(); i++)
    {
      for (uint j=0; j<atab_ind[i][MALE]; j++)
	{
	  const usint posx=aging_tab[i][MALE][j].first;
	  const usint posy=aging_tab[i][MALE][j].second;
          bool loser=false;
	  /*
	    if (env==NULL || env->predictable())
            {
	    if (tab.get(posx,posy).getDeathYear()<gennum) //or <=
	    loser=true;
            }
	    else*/
	  {
	    if (isLoser(tab.get(posx,posy),posx,posy,0,env))
	      loser=true;
	  }

          if (loser)
	    {
              if (tab.get(posx, posy).isFIVETed()) F_specimen_count--;
              tab.setAlive(posx,posy,false);
	      specimen_count--;
	      if(L.isPanmictic())
	        L.panPutFreePlace(posx,posy);
	      aging_tab[i][MALE][j]=aging_tab[i][MALE][--atab_ind[i][MALE]];
              if (tab.get(posx,posy).isInfertile())
                --atab_ind[i][INF_MALE];
              if (tab.get(posx,posy).isFIVETed())
                --atab_ind[i][FIVETed_MALE];
	      j--; //in order not to skip already checked position
	    }
	}

      for (uint j=0; j<atab_ind[i][FEMALE]; j++)
	{
	  const usint posx=aging_tab[i][FEMALE][j].first;
	  const usint posy=aging_tab[i][FEMALE][j].second;
          bool loser=false;
          /*
	    if (env==NULL || env->predictable())
            {
	    if (tab.get(posx,posy).getDeathYear()<gennum) //or <=
	    loser=true;
            }
            else*/
	  {
	    if (isLoser(tab.get(posx,posy),posx,posy,0,env))
	      loser=true;
	  }

	  if (loser)
	    {
              if (tab.get(posx, posy).isFIVETed()) F_specimen_count--;
              tab.setAlive(posx,posy,false);
	      specimen_count--;
	      if(L.isPanmictic())
	        L.panPutFreePlace(posx,posy);
	      aging_tab[i][FEMALE][j]=aging_tab[i][FEMALE][--atab_ind[i][FEMALE]];
              if (tab.get(posx,posy).isInfertile())
                --atab_ind[i][INF_FEMALE];
              if (tab.get(posx,posy).isFIVETed())
                --atab_ind[i][FIVETed_FEMALE];
	      j--;
	    }
	}
    }

  gennum++;
}

void Atab::updateIterators()
{
  vec.clear();

  //fill rand_vec with females in the reproduction age
  for (usint i=pop->getReproductionAge(); i<pop->getLifeSpan(); i++)
    for (uint j=0; j<atab_ind[i][FEMALE]; j++)
      vec.push_back(aging_tab[i][FEMALE][j]);
  begin_=vec.begin();
  end_=vec.end();
  //set number of males in the reproduction period
  males_in_reproduction_period_max=0;
  for (usint i=pop->getReproductionAge(); i<pop->getLifeSpan(); i++)
    {
      males_in_reproduction_period_max+=atab_ind[i][MALE];
      males_in_reproduction_period[i]=males_in_reproduction_period_max;
    }
}

uint Atab::getAtabInd(usint age, Gender g) const
{
  return atab_ind[age][g];
}

uint Atab::getSpecimenCount() const
{
  return getSpecimenCount(false);
}

uint Atab::getSpecimenCount(bool f) const
{
  if (!f) return specimen_count;
  return F_specimen_count;
}

Point Atab::getGuy(usint age, Gender g, uint pos) const
{
  return aging_tab[age][g][pos];
}

bool Atab::kill(Tab &tab, usint x, usint y, Gender g, uint age)
{
  for (uint i=0; i<atab_ind[age][g]; i++)
    if (aging_tab[age][g][i].first==x && aging_tab[age][g][i].second==y)
      {
	specimen_count--;
	aging_tab[age][g][i]=aging_tab[age][g][--atab_ind[age][g]];
        if (tab.get(x,y).isInfertile())
          --atab_ind[i][g+2];
        if (tab.get(x,y).isFIVETed())
          --atab_ind[i][g+4];
        iterators_invalidated=true;
	return true;
      }
  return false;
}

bool Atab::isLoserAbs(const Specimen & s, usint x, usint y, usint year, const Environment * env) const
{
  const usint age=year;

  assert(age<pop->getLifeSpan());

  uint bit_cnt=0,beg_of_chrom=0;

  const std::vector<uchar> & chrom_lengths=pop->getChromLengths();

  std::vector <uint> temp(pop->getNumChromosomes());

  //for every single chromosome
  for (usint j=0; j<pop->getNumChromosomes(); j++)
    {
      const uint factor=(8*sizeof(uint64_t)*chrom_lengths[j])/pop->getLifeSpan();
      const uint cut_no=(factor*age)/64+beg_of_chrom;
      const __uint64_t cut_mask=~mask64(63-(factor*age)%64);
      temp[j]=0;
      if (env==NULL) //no environment
        {
          for (uint i=beg_of_chrom; i<cut_no; i++)
            temp[j]+=bit_count64(s.h1[i]&s.h2[i]);

          temp[j]+=bit_count64( (cut_mask) & s.h1[cut_no] & s.h2[cut_no]);

          beg_of_chrom+=chrom_lengths[j];
        } else
        {
          for (uint i=beg_of_chrom; i<cut_no; i++)
            temp[j]+=bit_count64(s.h1[i]&s.h2[i]&(env->getEnv(i,gennum,x,y)));

          temp[j]+=bit_count64( (cut_mask) & s.h1[cut_no] & s.h2[cut_no] & (env->getEnv(cut_no,gennum,x,y)));

          beg_of_chrom+=chrom_lengths[j];
        }

      bit_cnt+=temp[j];
    }

  if (bit_cnt>=pop->getT())
    {
      if (age==pop->getBirthAge())
        {
	  for (usint j=0; j<pop->getNumChromosomes(); j++)
	    pop_stats.incrChromBA(j,temp[j]);
        } else
	if (age<=pop->getReproductionAge())
	  {
	    for (usint j=0; j<pop->getNumChromosomes(); j++)
	      pop_stats.incrChromBARA(j,temp[j]);
	  }
	else
	  {
	    for (usint j=0; j<pop->getNumChromosomes(); j++)
	      pop_stats.incrChromRA(j,temp[j]);
	  }

      return true;
    }

  return false;
}


bool Atab::isLoser(const Specimen & s, usint x, usint y, usint d_year, const Environment * env) const
{
  const usint age=s.getAge()+d_year;
  return isLoserAbs(s,x,y,age,env);
}

void Atab::atabInsert(usint y, usint x, const Specimen & s)
{
  uint age=s.getAge();
  Gender gender=s.getGender();
  assert(age<pop->getLifeSpan());
  assert(gender==MALE || gender==FEMALE);
  iterators_invalidated=true;

  if (atab_ind[age][gender]>=max_atab[age][gender])
    {
      max_atab[age][gender]+=100;

      Point * tmp=new Point[max_atab[age][gender]];

      for (uint i=0; i<atab_ind[age][gender]; i++)
	tmp[i]=aging_tab[age][gender][i];

      tmp[atab_ind[age][gender]].first=x;
      tmp[atab_ind[age][gender]].second=y;

      atab_ind[age][gender]++;
      if (s.isInfertile())
        atab_ind[age][gender+2]++;
      if (s.isFIVETed())
        atab_ind[age][gender+4]++;

      delete[] aging_tab[age][gender];

      aging_tab[age][gender]=tmp;
    } else
    {
      aging_tab[age][gender][atab_ind[age][gender]].first=x;
      aging_tab[age][gender][atab_ind[age][gender]].second=y;
      atab_ind[age][gender]++;
      if (s.isInfertile())
        atab_ind[age][gender+2]++;
      if (s.isFIVETed())
      {
        atab_ind[age][gender+4]++;
        F_specimen_count++;
      }
    }
  specimen_count++;
}



const Point Atab::getFather()
{
  //a new approach
  if (males_in_reproduction_period_max==0) return std::make_pair(0xffff,0xffff); // NO MALES FOUND

  uint male_number=rand()%males_in_reproduction_period_max;
  uint father_age=0;

  for (usint i=pop->getReproductionAge(); i<pop->getLifeSpan(); i++)
    if (males_in_reproduction_period[i]>male_number)
      {
	father_age=i;
	break;
      }

  uint temp=rand()%atab_ind[father_age][MALE]; //select one of MALES at this age

  return aging_tab[father_age][MALE][temp];
}

//not a really good code, because is run rarely
Atab::float2 Atab::computeFractionofDefects(const Tab &tab, usint chromosome) const
{
  const std::vector<uchar> & chrom_lengths=pop->getChromLengths();
  const uint factor=(8*sizeof(uint64_t)*chrom_lengths[chromosome])/pop->getLifeSpan();
  uint beg_of_chrom=0;

  for (uint i=0; i<chromosome; i++)
    beg_of_chrom+=chrom_lengths[i];

  const uint cut_no_BA=(factor*pop->getBirthAge())/64+beg_of_chrom;
  const uint cut_no_RA=(factor*pop->getReproductionAge())/64+beg_of_chrom;
  const __uint64_t cut_maskBA=~mask64(64-(factor*pop->getBirthAge())%64);
  const __uint64_t cut_maskRA=~mask64(64-(factor*pop->getReproductionAge())%64);


  uint number_of_individuals=0;
  uint64_t number_of_defectsBA=0;
  uint64_t number_of_defectsRA=0;

  for (usint i=1; i<pop->getLifeSpan(); i++)
    for (usint t=0; t<=1; t++)
      for (uint k=0; k<atab_ind[i][t]; k++)
	{
	  const usint posx=aging_tab[i][t][k].first;
	  const usint posy=aging_tab[i][t][k].second;
	  const Specimen & s=tab.get(posx,posy);

	  number_of_individuals++;

	  for (uint i2=beg_of_chrom; i2<cut_no_BA; i2++)
	    number_of_defectsBA+=bit_count64(s.h1[i2])+bit_count64(s.h2[i2]);

	  number_of_defectsBA+=bit_count64(cut_maskBA & s.h1[cut_no_BA]) + bit_count64(cut_maskBA & s.h2[cut_no_BA]);

	  number_of_defectsRA+=bit_count64((~cut_maskBA) & s.h1[cut_no_BA]) + bit_count64((~cut_maskBA) & s.h2[cut_no_BA]);

	  for (uint i2=cut_no_BA+1; i2<cut_no_RA; i2++)
	    number_of_defectsRA+=bit_count64(s.h1[i2])+bit_count64(s.h2[i2]);
	  number_of_defectsRA+=bit_count64(cut_maskRA & s.h1[cut_no_RA]) + bit_count64(cut_maskRA & s.h2[cut_no_RA]);
	}
  number_of_defectsBA/=2;
  number_of_defectsRA/=2;

  //return std::make_pair(number_of_defectsBA,number_of_defectsRA);
  //TODO:
  return std::make_pair(number_of_defectsBA/float(number_of_individuals*pop->getBirthAge()*factor),
  			number_of_defectsRA/float(number_of_individuals*(pop->getReproductionAge()-pop->getBirthAge())*factor));
}

void Atab::commitStats(const Tab &tab, uint gen) const
{
  std::vector<Atab::float2> data;

  for (uint i=0; i<pop->getNumChromosomes(); i++)
    data.push_back(computeFractionofDefects(tab,i));
  pop_stats.setFractions(data);


  for (usint i=pop->getBirthAge(); i<pop->getReproductionAge(); i++)
    {
      pop_stats.incrMaleBARA(atab_ind[i][MALE]);
      pop_stats.incrFemaleBARA(atab_ind[i][FEMALE]);
      pop_stats.incrInfMaleBARA(atab_ind[i][INF_MALE]);
      pop_stats.incrInfFemaleBARA(atab_ind[i][INF_FEMALE]);
    }

  for (usint i=pop->getReproductionAge(); i<pop->getLifeSpan(); i++)
    {
      pop_stats.incrMaleRA(atab_ind[i][MALE]);
      pop_stats.incrFemaleRA(atab_ind[i][FEMALE]);
      pop_stats.incrInfMaleRA(atab_ind[i][INF_MALE]);
      pop_stats.incrInfFemaleRA(atab_ind[i][INF_FEMALE]);
    }

  pop_stats.commit(gen);
}
