/*
 * Poisson.hpp
 *
 *  Created on: 2009-11-05
 *      Author: Wojciech Waga <wojciech.waga.com>
 */

#ifndef POISSON_HPP_
#define POISSON_HPP_

#include <stdexcept>
#include <vector>
#include <boost/math/distributions/poisson.hpp>
#include <boost/operators.hpp> //for equality_comparable
#include "NchrException.hpp"

class Poisson : private boost::equality_comparable<Poisson>
{
public:
  /**
   * @brief c-tor
   * @param m - number of chromosomes
   */
  Poisson(usint m):num_chrom(m),set(false)
  {//two allocations per one constructor - leak prone
    tab=new double*[num_chrom];
    size=new usint[num_chrom];
    for (usint i=0; i<num_chrom; i++)
      {
        size[i]=0;
        tab[i]=0;
      }
  }

  /**
   * @brief copy c-tor
   */
  Poisson(const Poisson &ref):num_chrom(ref.num_chrom),set(ref.set)
  {
    tab=new double*[num_chrom];
    size=new usint[num_chrom];
    if(set)
      {
	for (usint j=0; j<num_chrom; j++)
	  {
	    size[j]=ref.size[j];
	    tab[j]=new double[size[j]];
	    for (usint i=0; i<size[j]; i++)
	      tab[j][i]=ref.tab[j][i];
	  }
      }
    else
      for (usint i=0; i<num_chrom; i++)
        {
          size[i]=0;
          tab[i]=0;
        }
  }

  /**
   * @brief d-tor
   */
  ~Poisson()
  {
    delete[] size;
    for (usint i=0; i<num_chrom; i++)
      if(tab[i]) delete[] tab[i];
    delete[] tab;
  }

  /**
   * @brief assignment operator
   */
  Poisson & operator=(const Poisson &ref)
  {
    delete[] size;
    for (usint i=0; i<num_chrom; i++)
      if(tab[i]) delete[] tab[i];
    delete[] tab;

    num_chrom=ref.num_chrom;
    set=ref.set;

    tab=new double*[num_chrom];
    size=new usint[num_chrom];

    if(set)
      {
	for (usint j=0; j<num_chrom; j++)
	  {
	    size[j]=ref.size[j];
	    tab[j]=new double[size[j]];
	    for (usint i=0; i<size[j]; i++)
	      tab[j][i]=ref.tab[j][i];
	  }
      }
    else
      for (usint i=0; i<num_chrom; i++)
        {
          size[i]=0;
          tab[i]=0;
        }

    return *this;
  }

  /**
   * @brief set single lambda value for Poisson's distribution
   * @param f - float value to set
   */
  void setLambda(float f)
  {
    std::vector<float> v(1,f);
    setLambda(v);
  }

  /**
   * @brief set vector of lambdas for Poisson's distribuion for each chromosome
   * @param v - vector of floats, one per chromosome
   * @param factor - factor to multiply all values by (used for increades mutations)
   */
  void setLambda(std::vector<float> v, float factor=1)
  {
    if (v.size()!=num_chrom && v.size()!=1)
      throw NchrException("You have to specify as many lambdas as many chromosomes there are.");
    for (usint i=0; i<num_chrom; i++)
      if(tab[i]) delete[] tab[i];

    for (uint j=0; j<v.size(); j++)
      {
        std::vector<double> tmp;
        double sum=0;

        if (v[j]<1e-20) v[j]=1e-20; //to handle 0

        boost::math::poisson P(v[j]*factor);

        for (int i=0; i<(4*v[j]*factor+10); i++)
          {
            sum+=boost::math::pdf(P,i);
            tmp.push_back(sum); //prefix sum
            if (sum>0.9999 || i >= USHRT_MAX)
              break;
          }
        size[j]=(usint)tmp.size();
        tab[j]=new double[size[j]];
        for (usint i=0; i<size[j]; i++)
          tab[j][i]=tmp[i];
      }

    //if there is only one lambda specified copy probabilites from chr0 to all chrs
    if (v.size()==1)
      for (usint j=1; j<num_chrom; j++)
        {
          size[j]=size[0];
          tab[j]=new double[size[j]];
          for (usint i=0; i<size[j]; i++)
            tab[j][i]=tab[0][i];
        }
    set=true;
  }

  /**
   * @brief Get random number for a given chromosome
   * @param chr number of chromosome
   * @return random number
   */
  usint getNum(usint chromosome) const
  {
    assert(chromosome<num_chrom);
    if(!set)
      throw NchrException("Poisson not set, internal error");
    double value=rand_0to1();
    for (usint i=0; i<size[chromosome]; i++)
      if (value<tab[chromosome][i])
        return i;
    return size[chromosome]-1;
  }

  usint getNumChromosmes() const
  {
    return num_chrom;
  }

  bool operator==(const Poisson & p2) const
  {
    if (num_chrom!=p2.num_chrom) return false;
    if (set!=p2.set) return false;

    for (usint i=0; i<num_chrom; i++)
      {
        if (size[i]!=p2.size[i]) return false;
        for (usint j=0; j<size[i]; j++)
          if (fabs(tab[i][j]-p2.tab[i][j])>0.000001) return false;
      }
    return true;
  }

private:
  double **tab;
  usint *size;
  usint num_chrom;
  bool set;
};


#endif /* POISSON_HPP_ */
