/*
 * PopStatistics.hpp
 *
 *  Created on: 2010-02-09
 *      Author: Wojciech Waga <wojciech.waga.com>
 */

#ifndef POPSTATISTICS_HPP_
#define POPSTATISTICS_HPP_


#include <vector>
#include <boost/shared_ptr.hpp>
#include "../Population.hpp"

struct PopStatisticsChr
{
public:
  PopStatisticsChr():killingBeforeBA(0),killingBAtoRA(0),killingAfterRA(0),fractionBA(0),fractionBARA(0){}
  void clear()
  {
    killingBeforeBA=0;
    killingBAtoRA=0;
    killingAfterRA=0;
    fractionBA=0;
    fractionBARA=0;
  }

  uint64_t killingBeforeBA;
  uint64_t killingBAtoRA;
  uint64_t killingAfterRA;
  float fractionBA;
  float fractionBARA;
};

struct PopStatisticsGeneral
{
  PopStatisticsGeneral():num_males_BARA(0),num_males_RA(0),num_females_BARA(0),num_females_RA(0),infertile_num_males_BARA(0),infertile_num_males_RA(0),infertile_num_females_BARA(0),infertile_num_females_RA(0),conceptions(0),babies(0){}
  void clear()
  {
    num_males_BARA=0;
    num_males_RA=0;
    num_females_BARA=0;
    num_females_RA=0;
    infertile_num_males_BARA=0;
    infertile_num_males_RA=0;
    infertile_num_females_BARA=0;
    infertile_num_females_RA=0;
    conceptions=0;
    babies=0;
  }

  uint num_males_BARA;
  uint num_males_RA;
  uint num_females_BARA;
  uint num_females_RA;
  uint infertile_num_males_BARA;
  uint infertile_num_males_RA;
  uint infertile_num_females_BARA;
  uint infertile_num_females_RA;
  uint conceptions;
  uint babies;
};

struct LifeExpectancy
{
  LifeExpectancy(uint l):life_(l,0)
  {
  }

  void incr(size_t s)
  {
    life_[s]++;
  }

  std::vector<uint> life_;
};

class PopStatistics
{
public:
  typedef std::vector<PopStatisticsChr> Data1d;
  typedef std::vector<Data1d> Data2d;
  typedef std::vector<uint> Gen;
  typedef std::vector<PopStatisticsGeneral> General;

  PopStatistics(boost::shared_ptr<Population> p):chromosomes(p->getNumChromosomes()),temp(p->getNumChromosomes()),exp_(p->getLifeSpan())
  {
  }

  void incrChromBA(usint chr, uint val)
  {
    temp[chr].killingBeforeBA+=val;
  }

  void incrChromBARA(usint chr, uint val)
  {
    temp[chr].killingBAtoRA+=val;
  }

  void incrChromRA(usint chr, uint val)
  {
    temp[chr].killingAfterRA+=val;
  }

  void incrMaleBARA(uint num=1)
  {
    temp_non_chr.num_males_BARA+=num;
  }

  void incrMaleRA(uint num=1)
  {
    temp_non_chr.num_males_RA+=num;
  }

  void incrFemaleBARA(uint num=1)
  {
    temp_non_chr.num_females_BARA+=num;
  }

  void incrFemaleRA(uint num=1)
  {
    temp_non_chr.num_females_RA+=num;
  }

  void incrInfMaleBARA(uint num=1)
  {
    temp_non_chr.infertile_num_males_BARA+=num;
  }

  void incrInfMaleRA(uint num=1)
  {
    temp_non_chr.infertile_num_males_RA+=num;
  }

  void incrInfFemaleBARA(uint num=1)
  {
    temp_non_chr.infertile_num_females_BARA+=num;
  }

  void incrInfFemaleRA(uint num=1)
  {
    temp_non_chr.infertile_num_females_RA+=num;
  }

  void incrConception()
  {
    temp_non_chr.conceptions++;
  }

  void incrBirth()
  {
    temp_non_chr.babies++;
  }

  void incrLifeExpectancy(uint year)
  {
    exp_.incr(year);
  }

  void setFractions(const std::vector<std::pair<float,float> > &data);

  void commit(uint gen);

  const Data2d & getData() const
  {
    return chromosomes;
  }

  const General & getNonChrData() const
  {
    return non_chr_stats;
  }

  const Gen & getGennum() const
  {
    return gennum;
  }

  uint size() const
  {
    return chromosomes[0].size();
  }

  usint getNumChromosomes() const
  {
    return chromosomes.size();
  }

  const std::vector<uint> & getLifeExpectancy() const
  {
    return exp_.life_;
  }

private:
  Data2d chromosomes;
  Data1d temp;

  PopStatisticsGeneral temp_non_chr;
  General non_chr_stats;

  Gen gennum;

  LifeExpectancy exp_;
};


#endif /* POPSTATISTICS_HPP_ */
