/*
 * nchr2.cpp
 *
 *  Created on: 2009-12-31
 *      Author: Wojciech Waga <wojciech.waga.com>
 */

#include <fstream>
#include <iostream>
#include <vector>
#include <signal.h>
#include <boost/archive/binary_oarchive.hpp>
#include <boost/archive/binary_iarchive.hpp>
#include <boost/iostreams/filtering_stream.hpp>
#include <boost/iostreams/filter/bzip2.hpp>
#include <boost/python/suite/indexing/vector_indexing_suite.hpp>
#include "detail/defs.hpp"
#include "detail/NchrException.hpp"
#include "Environment.hpp"
#include "PopulationAPI.hpp"
#include "LatticeAPI.hpp"

#ifndef NCHRVERSION
#define NCHRVERSION "NO_VERSION";
#endif

using namespace boost::python;
namespace {

void sig_handler(int signum)
{
  if (signum==SIGUSR1)
    {
      std::cerr << '<';
      stats.print(std::cerr);
      std::cerr << '>';
      return;
    }
  std::cerr << "Signal " << signum << " caught. Trying to end up gracefully." << std::endl;
  if (sig_caught)
    {
      std::cerr << "...failed, you insisted too much!" << std::endl;
      std::cerr << "final_dump.nc is removed." << std::endl;
      remove("final_dump.nc2");
      exit(1);
    }
  ::sig_caught=signum;
}

void greet();

void echo(const std::string & text)
{
  std::cout << text;
}

void exits(uint return_code)
{
  exit(return_code);
}

}

namespace
{

  BOOST_PYTHON_MODULE(penna)
  {
    class_<Environment>("Env");
    class_<EnvSquare, bases<Environment> >("EnvSquare",init<int>());

    class_<Red, bases<Environment> >("Red",init<int>());
    class_<Green, bases<Environment> >("Green",init<int>());
    class_<Blue, bases<Environment> >("Blue",init<int>());
    class_<Plain, bases<Environment> >("Plain",init<>());

    docstring_options doc_options(true,true);
    doc_options.disable_signatures();

    def("echo",echo);
    def("exit",exits);

    def("bitcnt",bit_count64);

    class_<Specimen>("Specimen")
      .def("h1",&Specimen::getH1)
      .def("h2",&Specimen::getH2);

    class_<PopulationAPI>("Population",boost::python::init<const std::string &, const boost::python::list, bool>())
      .def("setFatherDist",&PopulationAPI::setFatherDist,arg("dist"),"Set distance of partner lookup.")
      .def("setChildDist",&PopulationAPI::setChildDist,arg("dist"),"Set distance of children placement.")
      .def("init",&PopulationAPI::initial,arg(""),"Init type.")
      .def("setB",&PopulationAPI::setB,arg("B"),"Set B.")
      .def("setT",&PopulationAPI::setT,arg("T"),"Set number of accepted defects in genome - T.")
      .def("setBirthAge",&PopulationAPI::setBirthAge,arg("birthAge"),"Set birth age for individuals.")
      .def("setReproductionAge",&PopulationAPI::setReproductionAge,arg("reprAge"),"Set reproduction age for individuals.")
      .def("setReconMarker",&PopulationAPI::setReconMarker, arg("chromosome, position"), "Set position of a marker used for gamete recognition.")
      .def("setRecombRate",(void (PopulationAPI::*)(float))(&PopulationAPI::setRecombRate),arg("r"),"Set recombination rate")
      .def("setRecombRate",(void (PopulationAPI::*)(boost::python::list))(&PopulationAPI::setRecombRate),arg("list"),"Set different recomb. rate for each chromosome.")
      .def("setMutationRate",(void (PopulationAPI::*)(float))(&PopulationAPI::setMutationRate),arg("r"),"Set Mutation rate.")
      .def("setMutationRate",(void (PopulationAPI::*)(boost::python::list))(&PopulationAPI::setMutationRate),arg("list"),"Set different mutation rate for each chromosome.")
      .def("setLifeSpan",&PopulationAPI::setLifeSpan,arg("int"),"Set maximum age that individuals can live.")
      .def("setMutationFactor",&PopulationAPI::setMutationFactor,arg("float"),"Set the number to multiply mutations by.")
      .def("setRecombinationFactor",&PopulationAPI::setRecombinationFactor,arg("float"),"Set the number to multiply recombinations by.")
      .def("setFivetFactor",&PopulationAPI::setFivetFactor,arg("float"),"Set the number to multiply mutation rate for in vitro conception.")
      .def("useFivet",&PopulationAPI::useFivet,arg("bool"),"Set whether or not population can use FIVET.")
      .def("hasFivet",&PopulationAPI::hasFivet,arg(""),"Whether or not population uses FIVET.")
      .def("setInfertility",&PopulationAPI::setInfertility,arg("float"),"Set the fraction of infertile individuals.")
      .def("setInfertilityAF",&PopulationAPI::setInfertilityAF,arg("float"),"Set the fraction of infertile individuals born with FIVET.")
      .def("setFivetEfficacy",&PopulationAPI::setFIVETefficacy,arg("float"),"Set the efficacy of FIVET (0..1).")
      .def("setPromisc",&PopulationAPI::setPromisc,arg("bool"),"Set promiscuous mode.")
      .def("getNumChromosomes",&PopulationAPI::getNumChromosomes,arg(""),"Get the bumber of chromosomes.")
      .def("getName",&PopulationAPI::getName,arg(""),"Get the name of population.")
      .def("getNumChunks",&PopulationAPI::getNumChunks,arg(""),"Get number of 64bit chunks of genome.")
      .def("getFatherDist",&PopulationAPI::getFatherDist,arg(""),"Get distance of partner lookup.")
      .def("getChildDist",&PopulationAPI::getChildDist,arg(""),"Get distance of children placement.")
      .def("getB",&PopulationAPI::getB,arg(""),"Get B.")
      .def("getT",&PopulationAPI::getT,arg(""),"Get T.")
      .def("getReproductionAge",&PopulationAPI::getReproductionAge,arg(""),"Get reproduction age for individuals.")
      .def("getRecombRate",&PopulationAPI::getRecombRate,arg(""),"Get recombination rate")
      .def("getMutationRate",&PopulationAPI::getMutationRate,arg(""),"Get recombination rate")
      .def("getBirthAge",&PopulationAPI::getBirthAge,arg(""),"Get birth age for individuals.");


    class_<LatticeAPI>("Lattice",boost::python::init<int, int>())
      .def(boost::python::init<std::string>())
      .def("snpPng",&LatticeAPI::snpPngA, arg("string"), "Draw a snapshot of the lattice.")
      .def("snpPng",&LatticeAPI::snpPngB, arg("string"), "Draw a snapshot of the lattice.")

      .def("snpPmg",&LatticeAPI::snpPop, arg("string"), "")

      .def("LDPng",&LatticeAPI::LDPng, arg("string"), "Draw an LD for the Lattice.")

      .def("chrPng",&LatticeAPI::chrPngA, arg("string"), "Draw chromosomes plot.")
      .def("chrPng",&LatticeAPI::chrPngB, arg("string"), "Draw chromosomes plot.")
      .def("chrPng",&LatticeAPI::chrPngC, arg("string"), "")
      .def("chrPng",&LatticeAPI::chrPngD, arg("string"), "")

      .def("chrTxt",&LatticeAPI::chrTxtA, arg("string"), "")
      .def("chrTxt",&LatticeAPI::chrTxtB, arg("string"), "")
      .def("chrTxt",&LatticeAPI::chrTxtC, arg("string"), "")
      .def("chrTxt",&LatticeAPI::chrTxtD, arg("string"), "")

      .def("agePng",&LatticeAPI::agePngA, arg("string"), "Draw age structure of the individuals.")
      .def("agePng",&LatticeAPI::agePngB, arg("string"), "Draw age structure of the individuals.")
      .def("agePng",&LatticeAPI::agePngC, arg("string"), "")
      .def("agePng",&LatticeAPI::agePngD, arg("string"), "")

      .def("ageTxt",&LatticeAPI::ageTxtA, arg("string"), "")
      .def("ageTxt",&LatticeAPI::ageTxtB, arg("string"), "")

      .def("statistics",&LatticeAPI::statistics, arg("string"), "Write ODS file with statistict for each population.")
      .def("statsPeriod",&LatticeAPI::setStatsPeriod,arg("generations"),"Set the number of generations for stats collection.")

      .def("printStats",&LatticeAPI::printStats,arg("population name"),"Prints statistics for life expectancy for population. "
           "Statistics are collected only when lattice is frozen")

      .def("popCnt",&LatticeAPI::popCntA, arg("string"), "Get the number of individuals of a given population.")
      .def("popCnt",&LatticeAPI::popCntB, arg("string"), "")
      .def("purge",&LatticeAPI::purge, arg("float"), "")
      //.def("getGuy",&LatticeAPI::getGuy, arg("usint,usint"), "")
      //.def("isEmpty",&LatticeAPI::isEmpty, arg("usint,usint"), "")

      //.def("getHeight",&LatticeAPI::getHeight, arg(""), "")
      //.def("getWidth",&LatticeAPI::getWidth, arg(""), "")

      //.def("setInfertility",&LatticeAPI::setInfertility, arg("float"), "Fraction of infertile fields (0..1)")

      .def("listPops",&LatticeAPI::listPopulations, arg(""), "List all registered populations.")
      .def("getPop",&LatticeAPI::getPopulation, arg(""), "Get Population reference to change some values.")
      .def("copy",&LatticeAPI::copy, arg("source lattice,src x, src y, w, h, dst x, dst y"), "")

      .def("setImg",(void (LatticeAPI::*)(const std::string &,int))(&LatticeAPI::setImg), arg("int"), "Set image parameter.")
      .def("setImg",(void (LatticeAPI::*)(const std::string &, const std::string &))(&LatticeAPI::setImg), arg("string"), "Set image parameter.")
      .def("setImg",(void (LatticeAPI::*)(const std::string &, const boost::python::list&))(&LatticeAPI::setImg), arg("list"), "Set image parameter.")

      .def("save",&LatticeAPI::save, arg("string"), "Save simulation to *.bz2 file.")
      .def("killSquare",&LatticeAPI::killSquare, arg("x,y,w,z"), "")
      .def("setEnvironment",&LatticeAPI::setEnvironment,arg("env"),"Set environment.")
      .def("unsetEnvironment",&LatticeAPI::unsetEnvironment,arg(""),"Unset environment.")
      .def("fill",&LatticeAPI::fill1, arg("pop"), "Insert Population on lattice.")
      .def("fill",&LatticeAPI::fill2, arg("pop,x,y,w,z"), "Insert Population on lattice.")
      .def("nexGen",&LatticeAPI::nexGen,arg(""),"Advance to the next generation.")
      .def("frozenGen",&LatticeAPI::frozenGen,arg(""),"Simulate new generation.")
      .def("getGennum",&LatticeAPI::getGennum,arg(""),"Get generation number.")
      .def("setPanmixia",&LatticeAPI::setPanmixia, arg("panmixia"),"Toggle panmictic behaviour.")
      .def("cmp",&LatticeAPI::operator==, arg(""),"Compare objects.");
  }
}

int main(int argc, char * argv[])
{
  signal(SIGINT, sig_handler);
  signal(SIGUSR1, sig_handler);
  signal(SIGTERM, sig_handler);
  srand(time(0));

  Py_Initialize();
  stats.on();

  if (PyImport_AppendInittab((char*)"penna", initpenna) == -1)
    throw NchrException("Failed to load penna module. Fatal Error! Or should I say \"Super duper fatal error!!!\"?");

  PyObject* main_module =  PyImport_AddModule("__main__");
  PyObject* main_dict = PyModule_GetDict(main_module);

  PyRun_SimpleString("from penna import * \n");

  greet();

  if (argc==2 || argc==3)
    {
      FILE * file_1=fopen(argv[1],"r");
      if (!file_1)
	throw NchrException("Error reading file.");
      std::cout << "Running file " << argv[1] << std::endl << std::endl;
      PyRun_File(file_1,argv[1],Py_file_input,main_dict,main_dict);
    }

  if (argc!=2)
    Py_Main(1,argv);

  Py_Finalize();

  stats.print(std::cerr);
  return 0;
}

namespace
{

void greet()
{
  PyRun_SimpleString("B= \"\\033[1;34m\"\n");
  PyRun_SimpleString("G= \"\\033[1;32m\"\n");
  PyRun_SimpleString("R=\"\\033[1;31m\"\n");
  PyRun_SimpleString("N=\"\\033[0m\"\n");

  PyRun_SimpleString("print \"+-----------------------------------------------------------------------+\"\n");
  PyRun_SimpleString("print \"|                                                          <adun@o2.pl> |\"\n");

  if (isatty(STDOUT_FILENO))
    {
      PyRun_SimpleString("print \"|                  \\033[1;34mSm\\033[1;31mORF\\033[1;34mland team proudly presents:\",N,\"                   |\"\n");
      PyRun_SimpleString("print \"|                                                 \",G,\"The Nchr 2\",N,\"        |\"\n");
    } else
    {
      PyRun_SimpleString("print \"|         SmORFland team proudly presents:                              |\"\n");
      PyRun_SimpleString("print \"|                                                      The Nchr 2       |\"\n");
    }

  char buf[90];
  strcpy(buf,"print \"|  ver:");
  strcat(buf,NCHRVERSION);

  for (int i=strlen(buf); i<79; i++)
    strcat(buf," ");
  strcat(buf,"|\"\n");
  PyRun_SimpleString(buf);
  PyRun_SimpleString("print \"+-----------------------------------------------------------------------+\"\n");
}

}
